package com.topcoder.direct.services.copilot.dao;

import java.util.Date;

/**
 * This interface represents a helper utility DAO. It provides methods for retrieving the number of bug races for specific contest, the latest bug resolution date for specific contest, the earnings of specific copilot user and contest IDs for specific copilot user and TC direct project.
 * 
 * Thread Safety:
 * Implementations of this interface must be thread safe.
 */
public interface UtilityDAO {
    /**
     * Retrieves the number of bugs for contest with the specified ID. Returns 0 if contest with the given ID doesn't exist.
     * 
     * Parameters:
     * contestId - the ID of the contest
     * 
     * Returns:
     * the retrieved contest bug count (not negative)
     * 
     * Throws:
     * IllegalArgumentException if contestId <= 0
     * CopilotDAOException if some other error occurred
     * @throws IllegalArgumentException if contestId <= 0
     * @throws CopilotDAOException if some other error occurred
     * @param contestId the ID of the contest
     * @return the retrieved contest bug count (not negative)
     */
    public int getContestBugCount(long contestId)throws CopilotDAOException;

    /**
     * Retrieves the earnings of the copilot with the specified user ID.
     * 
     * Parameters:
     * userId - the ID of the copilot user
     * 
     * Returns:
     * the retrieved copilot earnings (not negative)
     * 
     * Throws:
     * IllegalArgumentException if userId <= 0
     * EntityNotFoundException if user with the given ID doesn't exist in the persistence
     * CopilotDAOException if some other error occurred
     * @throws IllegalArgumentException if userId <= 0
     * @throws EntityNotFoundException if user with the given ID doesn't exist in the persistence
     * @throws CopilotDAOException if some other error occurred
     * @param userId the ID of the copilot user
     * @return the retrieved copilot earnings (not negative)
     */
    public double getCopilotEarnings(long userId)throws CopilotDAOException;

    /**
     * Retrieves the latest bug resolution date for the contest. Returns null if contest with the given ID doesn't exist.
     * 
     * Parameters:
     * contestId - the ID of the contest
     * 
     * Returns:
     * the latest bug resolution date for the contest or null if the contest has no bugs
     * 
     * Throws:
     * IllegalArgumentException if contestId <= 0
     * CopilotDAOException if some other error occurred
     * @throws IllegalArgumentException if contestId <= 0
     * @throws CopilotDAOException if some other error occurred
     * @param contestId the ID of the contest
     * @return the latest bug resolution date for the contest or null if the contest has no bugs
     */
    public Date getContestLatestBugResolutionDate(long contestId)throws CopilotDAOException;

    /**
     * Retrieves IDs of all contests for the specified copilot user and TC direct project. Returns an empty array if copilot user or TC direct project with the specified ID doesn't exist.
     * 
     * Parameters:
     * copilotUserId - the ID of the copilot user
     * tcDirectProjectId - the ID of the TC direct project
     * 
     * Returns:
     * the retrieved IDs of copilot project contests (not null)
     * 
     * Throws:
     * IllegalArgumentException if copilotUserId <= 0 or tcDirectProjectId <= 0
     * CopilotDAOException if some other error occurred
     * @throws IllegalArgumentException if copilotUserId <= 0 or tcDirectProjectId <= 0
     * @throws CopilotDAOException if some other error occurred
     * @param copilotUserId the ID of the copilot user
     * @param tcDirectProjectId the ID of the TC direct project
     * @return the retrieved IDs of copilot project contests (not null)
     */
    public long[] getCopilotProjectContests(long copilotUserId, long tcDirectProjectId)throws CopilotDAOException;
}

