package com.topcoder.direct.services.copilot.dao;

import java.util.List;

import com.topcoder.direct.services.copilot.model.IdentifiableEntity;

/**
 * This interface represents a generic DAO for managing entities in persistence. It defines methods for creating, updating, deleting and retrieving entities to/from persistence.
 * 
 * Generic Parameters:
 * T - the type of the entity to be managed by this DAO
 * 
 * Thread Safety:
 * Implementations of this interface must be thread safe when entities passed to them are used by the caller in thread safe manner.
 * @param <T> 
 */
public interface GenericDAO<T extends IdentifiableEntity> {
    /**
     * Creates the given entity in the persistence.
     * 
     * Parameters:
     * entity - the entity to be created in the persistence
     * 
     * Returns:
     * the generated entity ID
     * 
     * Throws:
     * IllegalArgumentException if entity is null
     * CopilotDAOException if some other error occurred
     * @throws IllegalArgumentException if entity is null
     * @throws CopilotDAOException if some other error occurred
     * @param entity the entity to be created in the persistence
     * @return the generated entity ID
     */
    public long create(T entity)throws CopilotDAOException;

    /**
     * Updates the given entity in the persistence.
     * 
     * Parameters:
     * entity - the entity to be updated in the persistence
     * 
     * Throws:
     * IllegalArgumentException if entity is null or entity.getId() <= 0
     * EntityNotFoundException if entity with the same ID cannot be found in the persistence
     * CopilotDAOException if some other error occurred
     * @throws IllegalArgumentException if entity is null or entity.getId() <= 0
     * @throws EntityNotFoundException if entity with the same ID cannot be found in the persistence
     * @throws CopilotDAOException if some other error occurred
     * @param entity the entity to be updated in the persistence
     */
    public void update(T entity)throws CopilotDAOException;

    /**
     * Deletes an entity with the given ID from the persistence.
     * 
     * Parameters:
     * entityId - the ID of the entity to be deleted
     * 
     * Throws:
     * IllegalArgumentException if entityId <= 0
     * EntityNotFoundException if entity with the given ID doesn't exist in the persistence
     * CopilotDAOException if some other error occurred
     * @throws IllegalArgumentException if entityId <= 0
     * @throws EntityNotFoundException if entity with the given ID doesn't exist in the persistence
     * @throws CopilotDAOException if some other error occurred
     * @param entityId the ID of the entity to be deleted
     */
    public void delete(long entityId)throws CopilotDAOException;

    /**
     * Retrieves an entity from persistence by its ID. Returns null if entity with the given ID is not found.
     * 
     * Parameters:
     * entityId - the ID of the entity to be retrieved
     * 
     * Returns:
     * the entity with the specified ID retrieved from the persistence or null if not found
     * 
     * Throws:
     * IllegalArgumentException if entityId <= 0
     * CopilotDAOException if some other error occurred
     * @throws IllegalArgumentException if entityId <= 0
     * @throws CopilotDAOException if some other error occurred
     * @param entityId the ID of the entity to be retrieved
     * @return the entity with the specified ID retrieved from the persistence or null if not found
     */
    public T retrieve(long entityId)throws CopilotDAOException;

    /**
     * Retrieves all entities from the persistence. Returns an empty list if no entities are found.
     * 
     * Returns:
     * the list of entities found in the persistence (not null, doesn't contain null)
     * 
     * Throws:
     * CopilotDAOException if any error occurred
     * @throws CopilotDAOException if any error occurred
     * @return the list of entities found in the persistence (not null, doesn't contain null)
     */
    public List<T> retrieveAll()throws CopilotDAOException;
}

