package com.topcoder.direct.services.copilot.dao;

import com.topcoder.util.errorhandling.ExceptionData;


/**
 * This exception is thrown by implementations of UtilityDAO and GenericDAO when entity with the given ID doesn't exist in the persistence.
 * 
 * Thread Safety:
 * This class is not thread safe because its base class is not thread safe.
 */
public class EntityNotFoundException extends CopilotDAOException {
    /**
     * The type name of the entity that cannot be found. Is initialized in the constructor and never changed after that. Can be any value. Has a getter.
     */
    private final String entityTypeName;

    /**
     * The ID of the entity that cannot be found. Is initialized in the constructor and never changed after that. Can be any value. Has a getter.
     */
    private final long entityId;

    /**
     * Creates a new instance of this exception with the given message, entity type name and entity ID.
     * 
     * Parameters:
     * message - the detailed error message of this exception
     * entityTypeName - the type name of the entity that cannot be found
     * entityId - the ID of the entity that cannot be found
     * 
     * Implementation Notes:
     * 1. super(message);
     * 2. this.entityTypeName = entityTypeName;
     * 3. this.entityId = entityId;
     * @param message the detailed error message of this exception
     * @param entityTypeName the type name of the entity that cannot be found
     * @param entityId the ID of the entity that cannot be found
     */
    public EntityNotFoundException(String message, String entityTypeName, long entityId) {
        super(message);
        this.entityTypeName=entityTypeName;
        this.entityId=entityId;
    }

    /**
     * Creates a new instance of this exception with the given message, cause, entity type name and entity ID.
     * 
     * Parameters:
     * message - the detailed error message of this exception
     * cause - the inner cause of this exception
     * entityTypeName - the type name of the entity that cannot be found
     * entityId - the ID of the entity that cannot be found
     * 
     * Implementation Notes:
     * 1. super(message, cause);
     * 2. this.entityTypeName = entityTypeName;
     * 3. this.entityId = entityId;
     * @param message the detailed error message of this exception
     * @param entityTypeName the type name of the entity that cannot be found
     * @param cause the inner cause of this exception
     * @param entityId the ID of the entity that cannot be found
     */
    public EntityNotFoundException(String message, Throwable cause, String entityTypeName, long entityId) {
        super(message, cause);
        this.entityTypeName = entityTypeName;
        this.entityId = entityId;
    }

    /**
     * Creates a new instance of this exception with the given message, exception data, entity type name and entity ID.
     * 
     * Parameters:
     * message - the detailed error message of this exception
     * data - the exception data
     * entityTypeName - the type name of the entity that cannot be found
     * entityId - the ID of the entity that cannot be found
     * 
     * Implementation Notes:
     * 1. super(message, data);
     * 2. this.entityTypeName = entityTypeName;
     * 3. this.entityId = entityId;
     * @param message the detailed error message of this exception
     * @param entityTypeName the type name of the entity that cannot be found
     * @param entityId the ID of the entity that cannot be found
     * @param data the exception data
     */
    public EntityNotFoundException(String message, ExceptionData data, String entityTypeName, long entityId) {
        super(message, data);
        this.entityTypeName = entityTypeName;
        this.entityId = entityId;
    }

    /**
     * Creates a new instance of this exception with the given message, cause, exception data, entity type name and entity ID.
     * 
     * Parameters:
     * message - the detailed error message of this exception
     * cause - the inner cause of this exception
     * data - the exception data
     * entityTypeName - the type name of the entity that cannot be found
     * entityId - the ID of the entity that cannot be found
     * 
     * Implementation Notes:
     * 1. super(message, cause, data);
     * 2. this.entityTypeName = entityTypeName;
     * 3. this.entityId = entityId;
     * @param message the detailed error message of this exception
     * @param entityTypeName the type name of the entity that cannot be found
     * @param cause the inner cause of this exception
     * @param entityId the ID of the entity that cannot be found
     * @param data the exception data
     */
    public EntityNotFoundException(String message, Throwable cause, ExceptionData data, String entityTypeName,
        long entityId) {
        super(message,cause,data);
        this.entityTypeName = entityTypeName;
        this.entityId = entityId;
    }

    /**
     * Retrieves the type name of the entity that cannot be found.
     * 
     * Returns:
     * the type name of the entity that cannot be found
     * @return the type name of the entity that cannot be found
     */
    public String getEntityTypeName() {
        return entityTypeName;
    }

    /**
     * Retrieves the ID of the entity that cannot be found.
     * 
     * Returns:
     * the ID of the entity that cannot be found
     * @return the ID of the entity that cannot be found
     */
    public long getEntityId() {
        return entityId;
    }
}

