/*
 * Copyright (C) 2011 TopCoder Inc., All Rights Reserved.
 */
package com.csstem.testing;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;

import junit.framework.TestCase;

import com.thoughtworks.selenium.Selenium;


/**
 * Functional test for CS-STEM parent module.
 *
 * @author TCSDEVELOPER
 * @version 1.0
 */
public class ParentFunctionalTests extends TestCase {
    /** Represent the thread sleep time. */
    private static final long SLEEP = 500;

    /**
     * Represents a 101 length string.
     */
	private static final String ONEZEROONE_STRING = "aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaB";

    /** Represents the Selenium Instance. */
    private Selenium browser;

    /**
     * Represents a 51 length string.
     */
    private final String FIFTYONE_STRING = "aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaab";

    /**
     * If a restore profile is required.
     */
    private boolean restoreProfile = false;

    /**
     * If a restore account is required.
     */
    private boolean restoreAccount = false;

    /**
     * Sets up the testing environment.
     *
     * @throws Exception if any error occurs.
     */
    public void setUp() throws Exception {
        browser = TestHelper.getIndexPage();
        super.setUp();
    }

    /**
     * Tears down the testing environment.
     *
     * @throws Exception if any error occurs.
     */
    public void tearDown() throws Exception {
    	if (restoreProfile) {
    		restoreProfile();
    	}
    	if (restoreAccount) {
    		restoreAccount();
    	}
        browser.stop();
        restoreProfile = false;
        restoreAccount = false;
        super.tearDown();
    }

    /**
     * FTC 1: Verify user's profile page can be displayed.
     * @throws Exception if any error
     */
    public void testFTC1() throws Exception {
        browser = TestHelper.getLoggedInAsParent(browser);
    	enterMyAccount();
		browser.click("//div[@id='noApproveTab']/div[1]/ul/li[5]/a/span/span");
		browser.waitForPageToLoad(TestHelper.getTimeout());
		assertEquals("the field value should be correct", "first", browser.getValue("_parentmyprofileportlet_WAR_parentregistrationportlet_firstName"));
		assertEquals("the field value should be correct", "middle", browser.getValue("_parentmyprofileportlet_WAR_parentregistrationportlet_middleName"));
		assertEquals("the field value should be correct", "last", browser.getValue("_parentmyprofileportlet_WAR_parentregistrationportlet_lastName"));
		assertEquals("the field value should be correct", "United States", browser.getValue("_parentmyprofileportlet_WAR_parentregistrationportlet_country"));
		assertEquals("the field value should be correct", "Alabama", browser.getValue("_parentmyprofileportlet_WAR_parentregistrationportlet_state"));
		assertEquals("the field value should be correct", "al", browser.getValue("_parentmyprofileportlet_WAR_parentregistrationportlet_city"));
		assertEquals("the field value should be correct", "28839", browser.getValue("_parentmyprofileportlet_WAR_parentregistrationportlet_zip"));
		assertEquals("the field value should be correct", "1970-02-01", browser.getValue("_parentmyprofileportlet_WAR_parentregistrationportlet_dateOfBirth"));
		assertEquals("the gender should be correct", "Male", browser.getSelectedLabel("gender"));
    }

    /**
     * FTC 2: Verify user's account management page can be displayed.
     * @throws Exception if any error
     */
    public void testFTC2() throws Exception {
        browser = TestHelper.getLoggedInAsParent(browser);
    	enterMyAccount();
		browser.click("//div[@id='noApproveTab']/div[1]/ul/li[4]/a/span/span");
		browser.waitForPageToLoad(TestHelper.getTimeout());
		assertEquals("the handle value should be uneditable", "disabled", browser.getAttribute("//div[@class='form accountManagementForm']/p[1]/input[1]@disabled"));
		assertEquals("the field value should be correct", "csstemford", browser.getValue("//div[@class='form accountManagementForm']/p[1]/input[1]"));
		assertEquals("the field value should be correct", "", browser.getValue("_parentaccountmanagementportlet_WAR_parentprofilepagesportlet_password"));
		assertEquals("the field value should be correct", "", browser.getValue("_parentaccountmanagementportlet_WAR_parentprofilepagesportlet_confirmPassword"));
		assertEquals("the field value should be correct", "csstem1@163.com", browser.getValue("_parentaccountmanagementportlet_WAR_parentprofilepagesportlet_email"));
    }

    /**
     * FTC 3: Verify user's email can be modified successfully.
     * @throws Exception if any error
     */
    public void testFTC3() throws Exception {
    	restoreAccount = true;
        browser = TestHelper.getLoggedInAsParent(browser);
    	enterMyAccount();
		browser.click("//div[@id='noApproveTab']/div[1]/ul/li[4]/a/span/span");
		browser.waitForPageToLoad(TestHelper.getTimeout());
		assertEquals("the field value should be correct", "csstem1@163.com", browser.getValue("_parentaccountmanagementportlet_WAR_parentprofilepagesportlet_email"));
		browser.type("_parentaccountmanagementportlet_WAR_parentprofilepagesportlet_email", "updated@test.com");
		browser.click("//a[@class='greenBtn saveButton']");
		browser.waitForPageToLoad(TestHelper.getTimeout());
		assertTrue("the field value should be updated", browser.isTextPresent("Account updated successfully!"));
		assertEquals("the field value should be correct", "updated@test.com", browser.getValue("_parentaccountmanagementportlet_WAR_parentprofilepagesportlet_email"));
    }

    /**
     * FTC 4: Verify form validation on account management page works well (empty email).
     * @throws Exception if any error
     */
    public void testFTC4() throws Exception {
    	restoreAccount = true;
        browser = TestHelper.getLoggedInAsParent(browser);
    	enterMyAccount();
		browser.click("//div[@id='noApproveTab']/div[1]/ul/li[4]/a/span/span");
		browser.waitForPageToLoad(TestHelper.getTimeout());
		assertEquals("the field value should be correct", "csstem1@163.com", browser.getValue("_parentaccountmanagementportlet_WAR_parentprofilepagesportlet_email"));
		browser.type("_parentaccountmanagementportlet_WAR_parentprofilepagesportlet_email", "");
		browser.click("//a[@class='greenBtn saveButton']");
		browser.waitForPageToLoad(TestHelper.getTimeout());
		assertTrue("the field is required", browser.isTextPresent("Field email is required!"));
    }

    /**
     * FTC 5: Verify form validation on account management page works well (invalid email).
     * @throws Exception if any error
     */
    public void testFTC5a() throws Exception {
    	restoreAccount = true;
        browser = TestHelper.getLoggedInAsParent(browser);
    	enterMyAccount();
		browser.click("//div[@id='noApproveTab']/div[1]/ul/li[4]/a/span/span");
		browser.waitForPageToLoad(TestHelper.getTimeout());
		assertEquals("the field value should be correct", "csstem1@163.com", browser.getValue("_parentaccountmanagementportlet_WAR_parentprofilepagesportlet_email"));
		browser.type("_parentaccountmanagementportlet_WAR_parentprofilepagesportlet_email", "user123#test.com");
		browser.click("//a[@class='greenBtn saveButton']");
		browser.waitForPageToLoad(TestHelper.getTimeout());
		assertTrue("the field value should be updated", browser.isTextPresent("Please enter a valid email."));
    }

    /**
     * FTC 5: Verify form validation on account management page works well (invalid email).
     * @throws Exception if any error
     */
    public void testFTC5b() throws Exception {
    	restoreAccount = true;
        browser = TestHelper.getLoggedInAsParent(browser);
    	enterMyAccount();
		browser.click("//div[@id='noApproveTab']/div[1]/ul/li[4]/a/span/span");
		browser.waitForPageToLoad(TestHelper.getTimeout());
		assertEquals("the field value should be correct", "csstem1@163.com", browser.getValue("_parentaccountmanagementportlet_WAR_parentprofilepagesportlet_email"));
		browser.type("_parentaccountmanagementportlet_WAR_parentprofilepagesportlet_email", "username12341234567890123456789012345678901234567890123456789012345678901234567890123456789012@test.com");
		browser.click("//a[@class='greenBtn saveButton']");
		browser.waitForPageToLoad(TestHelper.getTimeout());
		assertTrue("the field value should be updated", browser.isTextPresent("Email address can't be longer than 100 characters."));
    }

    /**
     * FTC 6: Verify form validation on account management page works well (empty password/confirm password).
     * @throws Exception if any error
     */
    public void testFTC6() throws Exception {
    	restoreAccount = true;
        browser = TestHelper.getLoggedInAsParent(browser);
    	enterMyAccount();
		browser.click("//div[@id='noApproveTab']/div[1]/ul/li[4]/a/span/span");
		browser.waitForPageToLoad(TestHelper.getTimeout());
		assertEquals("the field value should be correct", "csstem1@163.com", browser.getValue("_parentaccountmanagementportlet_WAR_parentprofilepagesportlet_email"));
		browser.type("_parentaccountmanagementportlet_WAR_parentprofilepagesportlet_email", "csstem1@test.com");
		browser.type("_parentaccountmanagementportlet_WAR_parentprofilepagesportlet_password", "");
		browser.type("_parentaccountmanagementportlet_WAR_parentprofilepagesportlet_confirmPassword", "");
		browser.click("//a[@class='greenBtn saveButton']");
		browser.waitForPageToLoad(TestHelper.getTimeout());
		assertTrue("the field value should be updated", browser.isTextPresent("Account updated successfully!"));
		assertTrue("the password/confirm password is not required", browser.isTextPresent("Leave passwords blank to keep current password"));
    }

    /**
     * FTC 7: Verify form validation on account management page works well (unmatched password and confirm password).
     * @throws Exception if any error
     */
    public void testFTC7() throws Exception {
    	restoreAccount = true;
        browser = TestHelper.getLoggedInAsParent(browser);
    	enterMyAccount();
		browser.click("//div[@id='noApproveTab']/div[1]/ul/li[4]/a/span/span");
		browser.waitForPageToLoad(TestHelper.getTimeout());
		assertEquals("the field value should be correct", "csstem1@163.com", browser.getValue("_parentaccountmanagementportlet_WAR_parentprofilepagesportlet_email"));
		browser.type("_parentaccountmanagementportlet_WAR_parentprofilepagesportlet_email", "csstem1@test.com");
		browser.type("_parentaccountmanagementportlet_WAR_parentprofilepagesportlet_password", "abcdef");
		browser.type("_parentaccountmanagementportlet_WAR_parentprofilepagesportlet_confirmPassword", "123456");
		browser.click("//a[@class='greenBtn saveButton']");
		browser.waitForPageToLoad(TestHelper.getTimeout());
		assertTrue("the password/confirm password is not required", browser.isTextPresent("Passwords do not match."));
    }

    /**
     * FTC 8: Verify user's password can be modified successfully.
     * @throws Exception if any error
     */
    public void testFTC8() throws Exception {
        browser = TestHelper.getLoggedInAsParent(browser);
    	enterMyAccount();
		browser.click("//div[@id='noApproveTab']/div[1]/ul/li[4]/a/span/span");
		browser.waitForPageToLoad(TestHelper.getTimeout());
		assertEquals("the field value should be correct", "csstem1@163.com", browser.getValue("_parentaccountmanagementportlet_WAR_parentprofilepagesportlet_email"));
		browser.type("_parentaccountmanagementportlet_WAR_parentprofilepagesportlet_email", "csstem1@test.com");
		browser.type("_parentaccountmanagementportlet_WAR_parentprofilepagesportlet_password", "123456");
		browser.type("_parentaccountmanagementportlet_WAR_parentprofilepagesportlet_confirmPassword", "123456");
		browser.click("//a[@class='greenBtn saveButton']");
		browser.waitForPageToLoad(TestHelper.getTimeout());
		assertTrue("the field value should be updated", browser.isTextPresent("Account updated successfully!"));
		browser.type("_parentaccountmanagementportlet_WAR_parentprofilepagesportlet_email", "csstem1@163.com");
		browser.type("_parentaccountmanagementportlet_WAR_parentprofilepagesportlet_password", "111111");
		browser.type("_parentaccountmanagementportlet_WAR_parentprofilepagesportlet_confirmPassword", "111111");
		browser.click("//a[@class='greenBtn saveButton']");
		browser.waitForPageToLoad(TestHelper.getTimeout());
    }

    /**
     * FTC 10: Verify user's profile can be updated from "My Profile" page.
     * @throws Exception if any error
     */
    public void testFTC10() throws Exception {
    	restoreProfile = true;
        browser = TestHelper.getLoggedInAsParent(browser);
    	enterMyAccount();
		browser.click("//div[@id='noApproveTab']/div[1]/ul/li[5]/a/span/span");
		browser.waitForPageToLoad(TestHelper.getTimeout());
		assertEquals("the field value should be correct", "first", browser.getValue("_parentmyprofileportlet_WAR_parentregistrationportlet_firstName"));
		assertEquals("the field value should be correct", "middle", browser.getValue("_parentmyprofileportlet_WAR_parentregistrationportlet_middleName"));
		assertEquals("the field value should be correct", "last", browser.getValue("_parentmyprofileportlet_WAR_parentregistrationportlet_lastName"));
		assertEquals("the field value should be correct", "United States", browser.getValue("_parentmyprofileportlet_WAR_parentregistrationportlet_country"));
		assertEquals("the field value should be correct", "Alabama", browser.getValue("_parentmyprofileportlet_WAR_parentregistrationportlet_state"));
		assertEquals("the field value should be correct", "al", browser.getValue("_parentmyprofileportlet_WAR_parentregistrationportlet_city"));
		assertEquals("the field value should be correct", "28839", browser.getValue("_parentmyprofileportlet_WAR_parentregistrationportlet_zip"));
		assertEquals("the field value should be correct", "1970-02-01", browser.getValue("_parentmyprofileportlet_WAR_parentregistrationportlet_dateOfBirth"));
		assertEquals("the gender should be correct", "Male", browser.getSelectedLabel("gender"));
		browser.type("_parentmyprofileportlet_WAR_parentregistrationportlet_firstName", "updatedfirst");
		browser.type("_parentmyprofileportlet_WAR_parentregistrationportlet_middleName", "updatedmiddle");
		browser.type("_parentmyprofileportlet_WAR_parentregistrationportlet_lastName", "updatedlast");
		browser.type("_parentmyprofileportlet_WAR_parentregistrationportlet_country", "China");
		browser.select("gender", "Female");
		browser.click("//a[@class='greenBtn saveButton']");
		browser.waitForPageToLoad(TestHelper.getTimeout());
		assertEquals("the field value should be correct", "updatedfirst", browser.getValue("_parentmyprofileportlet_WAR_parentregistrationportlet_firstName"));
		assertEquals("the field value should be correct", "updatedmiddle", browser.getValue("_parentmyprofileportlet_WAR_parentregistrationportlet_middleName"));
		assertEquals("the field value should be correct", "updatedlast", browser.getValue("_parentmyprofileportlet_WAR_parentregistrationportlet_lastName"));
		assertEquals("the field value should be correct", "China", browser.getValue("_parentmyprofileportlet_WAR_parentregistrationportlet_country"));
		assertEquals("the gender should be correct", "Female", browser.getSelectedLabel("gender"));
    }

    /**
     * FTC 11: Verify user's profile will not be updated if pressing "Cancel".
     * @throws Exception if any error
     */
    public void testFTC11() throws Exception {
        browser = TestHelper.getLoggedInAsParent(browser);
    	enterMyAccount();
		browser.click("//div[@id='noApproveTab']/div[1]/ul/li[5]/a/span/span");
		browser.waitForPageToLoad(TestHelper.getTimeout());
		assertEquals("the field value should be correct", "first", browser.getValue("_parentmyprofileportlet_WAR_parentregistrationportlet_firstName"));
		assertEquals("the field value should be correct", "middle", browser.getValue("_parentmyprofileportlet_WAR_parentregistrationportlet_middleName"));
		assertEquals("the field value should be correct", "last", browser.getValue("_parentmyprofileportlet_WAR_parentregistrationportlet_lastName"));
		assertEquals("the field value should be correct", "United States", browser.getValue("_parentmyprofileportlet_WAR_parentregistrationportlet_country"));
		assertEquals("the field value should be correct", "Alabama", browser.getValue("_parentmyprofileportlet_WAR_parentregistrationportlet_state"));
		assertEquals("the field value should be correct", "al", browser.getValue("_parentmyprofileportlet_WAR_parentregistrationportlet_city"));
		assertEquals("the field value should be correct", "28839", browser.getValue("_parentmyprofileportlet_WAR_parentregistrationportlet_zip"));
		assertEquals("the field value should be correct", "1970-02-01", browser.getValue("_parentmyprofileportlet_WAR_parentregistrationportlet_dateOfBirth"));
		assertEquals("the gender should be correct", "Male", browser.getSelectedLabel("gender"));
		browser.type("_parentmyprofileportlet_WAR_parentregistrationportlet_firstName", "updatedfirst");
		browser.type("_parentmyprofileportlet_WAR_parentregistrationportlet_middleName", "updatedmiddle");
		browser.type("_parentmyprofileportlet_WAR_parentregistrationportlet_lastName", "updatedlast");
		browser.type("_parentmyprofileportlet_WAR_parentregistrationportlet_country", "China");
		browser.select("gender", "Female");
		browser.click("//a[@class='grayBtn1 cancelButton']");
		Thread.sleep(SLEEP);
		assertEquals("the field value should be correct", "first", browser.getValue("_parentmyprofileportlet_WAR_parentregistrationportlet_firstName"));
		assertEquals("the field value should be correct", "middle", browser.getValue("_parentmyprofileportlet_WAR_parentregistrationportlet_middleName"));
		assertEquals("the field value should be correct", "last", browser.getValue("_parentmyprofileportlet_WAR_parentregistrationportlet_lastName"));
		assertEquals("the field value should be correct", "United States", browser.getValue("_parentmyprofileportlet_WAR_parentregistrationportlet_country"));
		assertEquals("the gender should be correct", "Male", browser.getSelectedLabel("gender"));
    }

    /**
     * FTC 12: Verify form validation works well on "My Profile" page (Invalid First Name).
     * @throws Exception if any error
     */
    public void testFTC12() throws Exception {
        browser = TestHelper.getLoggedInAsParent(browser);
    	enterMyAccount();
		browser.click("//div[@id='noApproveTab']/div[1]/ul/li[5]/a/span/span");
		browser.waitForPageToLoad(TestHelper.getTimeout());
		browser.type("_parentmyprofileportlet_WAR_parentregistrationportlet_firstName", " ");
		browser.click("//a[@class='greenBtn saveButton']");
		browser.waitForPageToLoad(TestHelper.getTimeout());
		assertTrue("error message is not shown", browser.isTextPresent("Field First Name is invalid"));
		browser.type("_parentmyprofileportlet_WAR_parentregistrationportlet_firstName", FIFTYONE_STRING);
		browser.click("//a[@class='greenBtn saveButton']");
		browser.waitForPageToLoad(TestHelper.getTimeout());
		assertTrue("error message is not shown", browser.isTextPresent("Field First Name is invalid"));
    }

    /**
     * FTC 13: Verify form validation works well on "My Profile" page (Invalid Last Name).
     * @throws Exception if any error
     */
    public void testFTC13() throws Exception {
        browser = TestHelper.getLoggedInAsParent(browser);
    	enterMyAccount();
		browser.click("//div[@id='noApproveTab']/div[1]/ul/li[5]/a/span/span");
		browser.waitForPageToLoad(TestHelper.getTimeout());
		browser.type("_parentmyprofileportlet_WAR_parentregistrationportlet_lastName", " ");
		browser.click("//a[@class='greenBtn saveButton']");
		browser.waitForPageToLoad(TestHelper.getTimeout());
		assertTrue("error message is not shown", browser.isTextPresent("Field Last Name is invalid"));
		browser.type("_parentmyprofileportlet_WAR_parentregistrationportlet_lastName", FIFTYONE_STRING);
		browser.click("//a[@class='greenBtn saveButton']");
		browser.waitForPageToLoad(TestHelper.getTimeout());
		assertTrue("error message is not shown", browser.isTextPresent("Field Last Name is invalid"));
    }

    /**
     * FTC 14: Verify form validation works well on "My Profile" page (Invalid City).
     * @throws Exception if any error
     */
    public void testFTC14() throws Exception {
        browser = TestHelper.getLoggedInAsParent(browser);
    	enterMyAccount();
		browser.click("//div[@id='noApproveTab']/div[1]/ul/li[5]/a/span/span");
		browser.waitForPageToLoad(TestHelper.getTimeout());
		browser.type("_parentmyprofileportlet_WAR_parentregistrationportlet_city", " ");
		browser.click("//a[@class='greenBtn saveButton']");
		browser.waitForPageToLoad(TestHelper.getTimeout());
		assertTrue("error message is not shown", browser.isTextPresent("Field City is invalid"));
		browser.type("_parentmyprofileportlet_WAR_parentregistrationportlet_city", ONEZEROONE_STRING);
		browser.click("//a[@class='greenBtn saveButton']");
		browser.waitForPageToLoad(TestHelper.getTimeout());
		assertTrue("error message is not shown", browser.isTextPresent("Field City is invalid"));
    }

    /**
     * FTC 14: Verify form validation works well on "My Profile" page (Invalid Zip).
     * @throws Exception if any error
     */
    public void testFTC15() throws Exception {
        browser = TestHelper.getLoggedInAsParent(browser);
    	enterMyAccount();
		browser.click("//div[@id='noApproveTab']/div[1]/ul/li[5]/a/span/span");
		browser.waitForPageToLoad(TestHelper.getTimeout());
		browser.type("_parentmyprofileportlet_WAR_parentregistrationportlet_zip", " ");
		browser.click("//a[@class='greenBtn saveButton']");
		browser.waitForPageToLoad(TestHelper.getTimeout());
		assertTrue("error message is not shown", browser.isTextPresent("Field Zip is invalid"));
		browser.type("_parentmyprofileportlet_WAR_parentregistrationportlet_zip", "1234");
		browser.click("//a[@class='greenBtn saveButton']");
		browser.waitForPageToLoad(TestHelper.getTimeout());
		assertTrue("error message is not shown", browser.isTextPresent("Field Zip is invalid"));
		browser.type("_parentmyprofileportlet_WAR_parentregistrationportlet_zip", "a2345");
		browser.click("//a[@class='greenBtn saveButton']");
		browser.waitForPageToLoad(TestHelper.getTimeout());
		assertTrue("error message is not shown", browser.isTextPresent("Field Zip is invalid"));
		browser.type("_parentmyprofileportlet_WAR_parentregistrationportlet_zip", "234558");
		browser.click("//a[@class='greenBtn saveButton']");
		browser.waitForPageToLoad(TestHelper.getTimeout());
		assertTrue("error message is not shown", browser.isTextPresent("Field Zip is invalid"));
    }

    /**
     * FTC 16: Verify form validation works well on "My Profile" page (Invalid Date of Birth).
     * @throws Exception if any error
     */
    public void testFTC16() throws Exception {
        browser = TestHelper.getLoggedInAsParent(browser);
    	enterMyAccount();
		browser.click("//div[@id='noApproveTab']/div[1]/ul/li[5]/a/span/span");
		browser.waitForPageToLoad(TestHelper.getTimeout());
		browser.type("_parentmyprofileportlet_WAR_parentregistrationportlet_dateOfBirth", " ");
		browser.click("//a[@class='greenBtn saveButton']");
		browser.waitForPageToLoad(TestHelper.getTimeout());
		assertTrue("error message is not shown", browser.isTextPresent("Field Date of Birth is invalid"));
		browser.type("_parentmyprofileportlet_WAR_parentregistrationportlet_dateOfBirth", "ab23-2a-3d");
		browser.click("//a[@class='greenBtn saveButton']");
		browser.waitForPageToLoad(TestHelper.getTimeout());
		assertTrue("error message is not shown", browser.isTextPresent("Field Date of Birth is invalid"));
		browser.type("_parentmyprofileportlet_WAR_parentregistrationportlet_dateOfBirth", "20111-11-11");
		browser.click("//a[@class='greenBtn saveButton']");
		browser.waitForPageToLoad(TestHelper.getTimeout());
		assertTrue("error message is not shown", browser.isTextPresent("Field Date of Birth is invalid"));
		browser.type("_parentmyprofileportlet_WAR_parentregistrationportlet_dateOfBirth", "1982-15-22");
		browser.click("//a[@class='greenBtn saveButton']");
		browser.waitForPageToLoad(TestHelper.getTimeout());
		assertTrue("error message is not shown", browser.isTextPresent("Field Date of Birth is invalid"));
		browser.type("_parentmyprofileportlet_WAR_parentregistrationportlet_dateOfBirth", "2050-05-05");
		browser.click("//a[@class='greenBtn saveButton']");
		browser.waitForPageToLoad(TestHelper.getTimeout());
		assertTrue("error message is not shown", browser.isTextPresent("Field Date of Birth is invalid"));
    }

    /**
     * FTC 18, 2.1.1.1: Verify the password is masked with asterisk.
     * @throws Exception if any error
     */
    public void testFTC18() throws Exception {
        browser = getRegistrationPage();
        assertEquals("the password type is wrong", "password", browser.getAttribute("_registrationportlet_WAR_studentregistrationportlet_password@type"));
    }

    /**
     * FTC 19, 2.1.1.1: Verify all the values in "Role" dropdown list are displayed.
     * @throws Exception if any error
     */
    public void testFTC19() throws Exception {
        browser = getRegistrationPage();
        assertTrue("the element is wrong", browser.isElementPresent("_registrationportlet_WAR_studentregistrationportlet_role"));
        String [] options = browser.getSelectOptions("_registrationportlet_WAR_studentregistrationportlet_role");
        assertEquals("the element is wrong", 4, options.length);
        List<String> list = new ArrayList<String>(Arrays.asList(options));
        assertTrue("the element is wrong", list.contains("Student"));
        assertTrue("the element is wrong", list.contains("Parent"));
        assertTrue("the element is wrong", list.contains("Teacher"));
        assertTrue("the element is wrong", list.contains("School Admin"));
    }

    /**
     * FTC 22, 2.1.1.1: Verify all the values in "Gender" dropdown list are displayed.
     * @throws Exception if any error
     */
    public void testFTC22() throws Exception {
        browser = getRegistrationPage();
        assertTrue("the element is wrong", browser.isElementPresent("_registrationportlet_WAR_studentregistrationportlet_gender"));
        String [] options = browser.getSelectOptions("_registrationportlet_WAR_studentregistrationportlet_gender");
        assertEquals("the element is wrong", 4, options.length);
        List<String> list = new ArrayList<String>(Arrays.asList(options));
        assertTrue("the element is wrong", list.contains("Male"));
        assertTrue("the element is wrong", list.contains("Female"));
        assertTrue("the element is wrong", list.contains("I decline to respond to this question"));
    }

    /**
     * FTC 23, 2.1.1.2: Verify "Terms & Conditions" can be clicked in the registration form.
     * @throws Exception if any error
     */
    public void testFTC23() throws Exception {
        browser = getRegistrationPage();
        assertEquals("only one window is displayed", 1, browser.getAllWindowIds().length);
        browser.click("link=Terms & Conditions");
        Thread.sleep(SLEEP);
        assertEquals("a popup window must be displayed", 2, browser.getAllWindowIds().length);
    }

    /**
     * FTC 24, 2.1.1.2: Verify "Terms & Conditions" can be clicked in the homepage footer.
     * @throws Exception if any error
     */
    public void testFTC24() throws Exception {
        browser.click("link=Terms and Conditions");
		browser.waitForPageToLoad(TestHelper.getTimeout());
        assertTrue("a popup window must be displayed", browser.isTextPresent("Terms & Conditions of Use at NoNameSite.com"));
        assertTrue("a popup window must be displayed", browser.isTextPresent("Acceptance of Terms and Conditions"));
    }

    /**
     * FTC 25, 2.1.1.1/2.1.1.2/2.1.1.4/2.1.1.5: Verify registration form validation works well (invalid handle).
     * @throws Exception if any error
     */
    public void testFTC25() throws Exception {
        browser = getRegistrationPage();
        browser.type("_registrationportlet_WAR_studentregistrationportlet_handle", "");
        browser.type("_registrationportlet_WAR_studentregistrationportlet_password", "123456");
        browser.type("profilConfirm", "123456");
        browser.type("_registrationportlet_WAR_studentregistrationportlet_email", "abc@abc.com");
        browser.select("_registrationportlet_WAR_studentregistrationportlet_role", "Parent");
        browser.select("_registrationportlet_WAR_studentregistrationportlet_gender", "Male");
        browser.select("suffix", "Washington");
        browser.check("_registrationportlet_WAR_studentregistrationportlet_agreeToTermsOfUse");
        browser.click("//a[@class='button greenBtn btnRegiter']");
        browser.waitForPageToLoad(TestHelper.getTimeout());
        assertTrue("error message is not displayed", browser.isTextPresent("Please enter a valid handle (4 to 8 characters, no spaces and numbers)"));
        browser.type("_registrationportlet_WAR_studentregistrationportlet_handle", "");
        browser.click("//a[@class='button greenBtn btnRegiter']");
        browser.waitForPageToLoad(TestHelper.getTimeout());
        assertTrue("error message is not displayed", browser.isTextPresent("Please enter a valid handle (4 to 8 characters, no spaces and numbers)"));
        browser.type("_registrationportlet_WAR_studentregistrationportlet_handle", "usr");
        browser.click("//a[@class='button greenBtn btnRegiter']");
        browser.waitForPageToLoad(TestHelper.getTimeout());
        assertTrue("error message is not displayed", browser.isTextPresent("Please enter a valid handle (4 to 8 characters, no spaces and numbers)"));
        browser.type("_registrationportlet_WAR_studentregistrationportlet_handle", "a bcd");
        browser.click("//a[@class='button greenBtn btnRegiter']");
        browser.waitForPageToLoad(TestHelper.getTimeout());
        assertTrue("error message is not displayed", browser.isTextPresent("Please enter a valid handle (4 to 8 characters, no spaces and numbers)"));
        browser.type("_registrationportlet_WAR_studentregistrationportlet_handle", "user12345");
        browser.click("//a[@class='button greenBtn btnRegiter']");
        browser.waitForPageToLoad(TestHelper.getTimeout());
        assertTrue("error message is not displayed", browser.isTextPresent("Please enter a valid handle (4 to 8 characters, no spaces and numbers)"));
        browser.type("_registrationportlet_WAR_studentregistrationportlet_handle", "%abcd");
        browser.click("//a[@class='button greenBtn btnRegiter']");
        browser.waitForPageToLoad(TestHelper.getTimeout());
        assertTrue("error message is not displayed", browser.isTextPresent("Please enter a valid handle (4 to 8 characters, no spaces and numbers)"));
    }

    /**
     * FTC 27, 2.1.1.1/2.1.1.2/2.1.1.4/2.1.1.5: Verify registration form validation works well (invalid password).
     * @throws Exception if any error
     */
    public void testFTC27() throws Exception {
        browser = getRegistrationPage();
        browser.type("_registrationportlet_WAR_studentregistrationportlet_handle", "testbb");
        browser.type("_registrationportlet_WAR_studentregistrationportlet_password", "");
        browser.type("profilConfirm", "123456");
        browser.type("_registrationportlet_WAR_studentregistrationportlet_email", "abc@abc.com");
        browser.select("_registrationportlet_WAR_studentregistrationportlet_role", "Parent");
        browser.select("_registrationportlet_WAR_studentregistrationportlet_gender", "Male");
        browser.select("suffix", "Washington");
        browser.check("_registrationportlet_WAR_studentregistrationportlet_agreeToTermsOfUse");
        browser.click("//a[@class='button greenBtn btnRegiter']");
        browser.waitForPageToLoad(TestHelper.getTimeout());
        assertTrue("error message is not displayed", browser.isTextPresent("Please enter a password."));
    }

    /**
     * FTC 28, 2.1.1.1/2.1.1.2/2.1.1.4/2.1.1.5: Verify registration form validation works well (invalid email).
     * @throws Exception if any error
     */
    public void testFTC28() throws Exception {
        browser = getRegistrationPage();
        browser.type("_registrationportlet_WAR_studentregistrationportlet_handle", "testbb");
        browser.type("_registrationportlet_WAR_studentregistrationportlet_password", "123456");
        browser.type("profilConfirm", "123456");
        browser.type("_registrationportlet_WAR_studentregistrationportlet_email", "");
        browser.select("_registrationportlet_WAR_studentregistrationportlet_role", "Parent");
        browser.select("_registrationportlet_WAR_studentregistrationportlet_gender", "Male");
        browser.select("suffix", "Washington");
        browser.check("_registrationportlet_WAR_studentregistrationportlet_agreeToTermsOfUse");
        browser.click("//a[@class='button greenBtn btnRegiter']");
        browser.waitForPageToLoad(TestHelper.getTimeout());
        assertTrue("error message is not displayed", browser.isTextPresent("Please enter a valid email."));
        browser.type("_registrationportlet_WAR_studentregistrationportlet_email", "user123#test.com");
        browser.click("//a[@class='button greenBtn btnRegiter']");
        browser.waitForPageToLoad(TestHelper.getTimeout());
        assertTrue("error message is not displayed", browser.isTextPresent("Please enter a valid email."));
        browser.type("_registrationportlet_WAR_studentregistrationportlet_email", "username12341234567890123456789012345678901234567890123456789012345678901234567890123456789012@test.com");
        browser.click("//a[@class='button greenBtn btnRegiter']");
        browser.waitForPageToLoad(TestHelper.getTimeout());
        assertTrue("error message is not displayed", browser.isTextPresent("Email address can't be longer than 100 characters."));
    }

    /**
     * FTC 29, 2.1.1.1/2.1.1.2/2.1.1.4/2.1.1.5: Verify registration form validation works well (invalid email).
     * @throws Exception if any error
     */
    public void testFTC29() throws Exception {
        browser = getRegistrationPage();
        List<String> options = new ArrayList<String>(Arrays.asList(browser.getSelectOptions("_registrationportlet_WAR_studentregistrationportlet_role")));
        assertEquals("the options are incorrect", 4, options.size());
        assertTrue("the options are incorrect", options.contains("Student"));
        assertTrue("the options are incorrect", options.contains("Parent"));
        assertTrue("the options are incorrect", options.contains("School Admin"));
        assertTrue("the options are incorrect", options.contains("Teacher"));
    }

    /**
     * FTC 30, 2.1.1.1/2.1.1.2/2.1.1.4/2.1.1.5: Verify registration form validation works well (Terms & Conditions checkbox not checked).
     * @throws Exception if any error
     */
    public void testFTC30() throws Exception {
        browser = getRegistrationPage();
        browser.type("_registrationportlet_WAR_studentregistrationportlet_handle", "testbb");
        browser.type("_registrationportlet_WAR_studentregistrationportlet_password", "123456");
        browser.type("profilConfirm", "123456");
        browser.type("_registrationportlet_WAR_studentregistrationportlet_email", "aaa@abc.com");
        browser.select("_registrationportlet_WAR_studentregistrationportlet_role", "Parent");
        browser.select("_registrationportlet_WAR_studentregistrationportlet_gender", "Male");
        browser.select("suffix", "Washington");
        browser.click("//a[@class='button greenBtn btnRegiter']");
        browser.waitForPageToLoad(TestHelper.getTimeout());
        assertTrue("error message is not displayed", browser.isTextPresent("You must agree to the terms and conditions of this application."));
    }

    /**
     * FTC 31, 2.1.1.1/2.1.1.2/2.1.1.4/2.1.1.5: Verify email is mandatory to register a new account.
     * @throws Exception if any error
     */
    public void testFTC31() throws Exception {
        browser = getRegistrationPage();
        browser.type("_registrationportlet_WAR_studentregistrationportlet_handle", "testbb");
        browser.type("_registrationportlet_WAR_studentregistrationportlet_password", "123456");
        browser.type("profilConfirm", "123456");
        browser.type("_registrationportlet_WAR_studentregistrationportlet_email", "");
        browser.select("_registrationportlet_WAR_studentregistrationportlet_role", "Parent");
        browser.select("_registrationportlet_WAR_studentregistrationportlet_gender", "Male");
        browser.select("suffix", "Washington");
        browser.check("_registrationportlet_WAR_studentregistrationportlet_agreeToTermsOfUse");
        browser.click("//a[@class='button greenBtn btnRegiter']");
        browser.waitForPageToLoad(TestHelper.getTimeout());
        assertTrue("error message is not displayed", browser.isTextPresent("Please enter a valid email."));
    }

    /**
     * FTC 38 2.2.1.1-2.2.1.4: Verify student will not be unregistered from the application if answering "No".
     * @throws Exception if any error
     */
    public void testFTC38() throws Exception {
        browser = TestHelper.getLoggedInAsParent(browser);
    	enterMyAccount();
        browser.click("//a[@title='Unregister Profile']");
		Thread.sleep(SLEEP);
        assertTrue("delete popup should be displayed", browser.isVisible("//div[@id='unregisterProfile']"));
		assertTrue("delete popup should be displayed", browser.isTextPresent("Unregister Profile"));
		assertTrue("delete popup should be displayed", browser.isTextPresent("Are you sure to unregister from CS-STEM application?"));
		assertTrue("delete popup should be displayed", browser.isTextPresent("Please note, it will be no more possible to register to this application without a special approval from our managers."));
		browser.runScript(";");
		Thread.sleep(SLEEP);
		assertTrue("user is not logged", browser.isTextPresent("Welcome csstemford"));
    }

    /**
     * FTC 39 2.2.1.1-2.2.1.4: Verify un-registration survey will be shown to authorized user.
     * @throws Exception if any error
     */
    public void testFTC39() throws Exception {
        browser.click("link=Login");
        browser.type("userNameInput", TestHelper.getParent3Username());
        browser.type("passwordInput", TestHelper.getParent3Password());
        browser.click("//div[@id='loginBoxC']/div[1]/a[1]/span/span");
        browser.waitForPageToLoad(TestHelper.getTimeout());
    	enterMyAccount();
		browser.click("//a[@class='greenBtn button unregisterProfileButton']");
		Thread.sleep(SLEEP);
        assertTrue("delete popup should be displayed", browser.isVisible("//div[@id='unregisterProfile']"));
		browser.click("//a[@class='greenBtn button']");
		browser.waitForPageToLoad(TestHelper.getTimeout());
		assertTrue("parent exit survey must be shown", browser.isTextPresent("Parent Exit Survey"));
		assertTrue("parent exit survey must be shown", browser.isTextPresent("Please help us improve CS STEM by answering the questions below."));
    }

    /**
     * FTC 40 2.2.1.1-2.2.1.4: Verify un-registration survey will not be submitted if pressing "Cancel".
     * @throws Exception if any error
     */
    public void testFTC40() throws Exception {
        browser.click("link=Login");
        browser.type("userNameInput", TestHelper.getParent3Username());
        browser.type("passwordInput", TestHelper.getParent3Password());
        browser.click("//div[@id='loginBoxC']/div[1]/a[1]/span/span");
        browser.waitForPageToLoad(TestHelper.getTimeout());
    	enterMyAccount();
		browser.click("//a[@class='greenBtn button unregisterProfileButton']");
		Thread.sleep(SLEEP);
        assertTrue("delete popup should be displayed", browser.isVisible("//div[@id='unregisterProfile']"));
		browser.click("//a[@class='greenBtn button']");
		browser.waitForPageToLoad(TestHelper.getTimeout());
		assertTrue("parent exit survey must be shown", browser.isTextPresent("Parent Exit Survey"));
		assertTrue("parent exit survey must be shown", browser.isTextPresent("Please help us improve CS STEM by answering the questions below."));
		browser.click("//a[@class='grayBtn button']");
		browser.waitForPageToLoad(TestHelper.getTimeout());
		assertTrue("user should be redirected to my account page", browser.isTextPresent("My Account"));
		assertEquals("user should be redirected to my account page", "NoNameSite.com | Parent Account", browser.getTitle());
    }

    /**
     * FTC 41 2.2.1.1-2.2.1.6: Verify validation for un-registration survey works well (radio buttons).
     * @throws Exception if any error
     */
    public void testFTC41() throws Exception {
        browser.click("link=Login");
        browser.type("userNameInput", TestHelper.getParent2Username());
        browser.type("passwordInput", TestHelper.getParent2Password());
        browser.click("//div[@id='loginBoxC']/div[1]/a[1]/span/span");
        browser.waitForPageToLoad(TestHelper.getTimeout());
    	enterMyAccount();
		browser.click("//a[@class='greenBtn button unregisterProfileButton']");
		Thread.sleep(SLEEP);
		browser.click("//a[@class='greenBtn button']");
		browser.waitForPageToLoad(TestHelper.getTimeout());
		assertTrue("parent exit survey must be shown", browser.isTextPresent("Parent Exit Survey"));
		assertTrue("parent exit survey must be shown", browser.isTextPresent("Please help us improve CS STEM by answering the questions below."));
		browser.click("//a[@title='Submit']");
		browser.waitForPageToLoad(TestHelper.getTimeout());
		assertTrue("submitting is not rejected", browser.isTextPresent("All the questions must be answered."));
    }

    /**
     * FTC 43 2.2.1.1-2.2.1.4: Verify un-registration survey will not be shown to unauthorized user
     * @throws Exception if any error
     */
    public void testFTC43() throws Exception {
		fail("once a parent is registered, he is already authorized");
    }

    /**
     * FTC 44 2.2.1.1-2.2.1.10: Verify parent will be informed on file about child student's unregistration.
     * @throws Exception if any error
     */
    public void testFTC44() throws Exception {
        browser = TestHelper.getLoggedInAsParent(browser);
    	enterMyAccount();
		browser.click("//div[@id='noApproveTab']/div[1]/ul/li[5]/a/span/span");
		browser.waitForPageToLoad(TestHelper.getTimeout());
		fail("no information for unregistered child");
    }

    /**
     * FTC 44 2.2.1.12: Verify parent's unregistration will lead to student's unauthorization.
     * @throws Exception if any error
     */
    public void testFTC46() throws Exception {
		fail("the user can't unregister due to validation error");
    }

    /**
     * FTC 55 2.5.1.1: Verify unauthorized student gets restricted to most activities.
     * @throws Exception if any error
     */
    public void testFTC55() throws Exception {
        browser.click("link=Login");
        browser.type("userNameInput", TestHelper.getStudentUsername());
        browser.type("passwordInput", TestHelper.getStudentPassword());
        browser.click("//div[@id='loginBoxC']/div[1]/a[1]/span/span");
        browser.waitForPageToLoad(TestHelper.getTimeout());
    	enterMyAccount();
		assertTrue("user is not limited access", browser.isTextPresent("You have to be authorized by your parent before receiving any recommended activity."));

		browser.open(TestHelper.getIndex());
		browser.click("//div[@id='p_p_id_activitieshome_WAR_activitiesportlet_INSTANCE_1moJ_']/div/div[1]/div[1]/div[4]/div/dl/dd[3]/a");
		browser.waitForPageToLoad(TestHelper.getTimeout());
		browser.click("//div[@id='p_p_id_activitylandingpagebuttonsportlet_WAR_contesttermsportlet_INSTANCE_3hDt_']/div/div/div/a[2]/span/span");
		browser.waitForPageToLoad(TestHelper.getTimeout());
		assertTrue("user is not limited access", browser.isTextPresent("Account Not Authorized"));
		assertTrue("user is not limited access", browser.isTextPresent("In order to participate in an activity, your account must be authorized. Please, try again after your parents sign your registration consent."));
    }

    /**
     * FTC 56 2.5.1.2: Verify authorized student gets access to most activities.
     * @throws Exception if any error
     */
    public void testFTC56() throws Exception {
    	browser = TestHelper.getLoggedInAsStudent(browser);
    	enterMyAccount();
		assertFalse("user is limited", browser.isTextPresent("You have to be authorized by your parent before receiving any recommended activity."));
		browser.click("link=Activity Permissions");
		browser.waitForPageToLoad(TestHelper.getTimeout());
		assertTrue("user should have permission", browser.isTextPresent("Showing 1-3 of 3 activities"));
		browser.click("link=Parent Contact Data");
		browser.waitForPageToLoad(TestHelper.getTimeout());
		browser.click("link=Manage Schools");
		browser.waitForPageToLoad(TestHelper.getTimeout());
		browser.click("link=Manage Teachers");
		browser.waitForPageToLoad(TestHelper.getTimeout());
		browser.click("link=Account Management");
		browser.waitForPageToLoad(TestHelper.getTimeout());
		browser.click("link=My Profile");
		browser.waitForPageToLoad(TestHelper.getTimeout());

		browser.open(TestHelper.getIndex());
		browser.click("//div[@id='p_p_id_activitieshome_WAR_activitiesportlet_INSTANCE_1moJ_']/div/div[1]/div[1]/div[4]/div/dl/dd[3]/a");
		browser.waitForPageToLoad(TestHelper.getTimeout());
		browser.click("//div[@id='p_p_id_activitylandingpagebuttonsportlet_WAR_contesttermsportlet_INSTANCE_3hDt_']/div/div/div/a[2]/span/span");
		browser.waitForPageToLoad(TestHelper.getTimeout());
		assertFalse("user is limited access", browser.isTextPresent("Account Not Authorized"));
		assertFalse("user is limited access", browser.isTextPresent("In order to participate in an activity, your account must be authorized. Please, try again after your parents sign your registration consent."));
    }

    /**
     * FTC 57 2.5.1.3: Verify student access to activities is controlled by parents.
     * @throws Exception if any error
     */
    public void testFTC57() throws Exception {
    	browser = TestHelper.getLoggedInAsStudent(browser);
    	enterMyAccount();
		assertFalse("user is limited", browser.isTextPresent("You have to be authorized by your parent before receiving any recommended activity."));
		browser.click("link=Activity Permissions");
		browser.waitForPageToLoad(TestHelper.getTimeout());
		assertTrue("permission should be controlled by parent", browser.isElementPresent("//a[@title='Request Permission']"));
		assertEquals("permission should be correct", "Allowed", browser.getText("//table[@class='generalTable generalListTable']/tbody[1]/tr[2]/td[5]"));
		assertEquals("permission should be correct", "Restricted", browser.getText("//table[@class='generalTable generalListTable']/tbody[1]/tr[3]/td[5]"));
		assertEquals("permission should be correct", "Allowed", browser.getText("//table[@class='generalTable generalListTable']/tbody[1]/tr[4]/td[5]"));
		browser.click("link=Brando Egg Hunter");
		browser.waitForPageToLoad(TestHelper.getTimeout());
		assertTrue("user is not limited access", browser.isTextPresent("You currently do not have the permission to join this activity. "));
		assertTrue("user is not limited access", browser.isTextPresent("Please request your parent's permission or try out the practice room for the time being. "));
    }

    /**
     * FTC 58 2.5.1.4: Verify student's permissions on the profile can be viewed by the student.
     * @throws Exception if any error
     */
    public void testFTC58() throws Exception {
    	browser = TestHelper.getLoggedInAsStudent(browser);
    	enterMyAccount();
		assertFalse("user is limited", browser.isTextPresent("You have to be authorized by your parent before receiving any recommended activity."));
		browser.click("link=Activity Permissions");
		browser.waitForPageToLoad(TestHelper.getTimeout());
		assertEquals("permission should be correct", "Allowed", browser.getText("//table[@class='generalTable generalListTable']/tbody[1]/tr[2]/td[5]"));
		assertEquals("permission should be correct", "Restricted", browser.getText("//table[@class='generalTable generalListTable']/tbody[1]/tr[3]/td[5]"));
		assertEquals("permission should be correct", "Allowed", browser.getText("//table[@class='generalTable generalListTable']/tbody[1]/tr[4]/td[5]"));
    }

    /**
     * FTC 63 2.7.1.3/2.7.1.4: Verify Student's Approval/Rejection Status on the Profile can be viewed.
     * @throws Exception if any error
     */
    public void testFTC63() throws Exception {
        browser.click("link=Login");
        browser.type("userNameInput", TestHelper.getAuthorizedStudentUsername());
        browser.type("passwordInput", TestHelper.getAuthorizedStudentPassword());
        browser.click("//div[@id='loginBoxC']/div[1]/a[1]/span/span");
        browser.waitForPageToLoad(TestHelper.getTimeout());
    	enterMyAccount();
		browser.click("link=My Profile");
		browser.waitForPageToLoad(TestHelper.getTimeout());
		fail("there is no approval/rejection status");
    }

    /**
     * FTC 64 2.7.1.5: Verify authorized student can also get restricted to activities by parents control.
     * @throws Exception if any error
     */
    public void testFTC64() throws Exception {
        browser.click("link=Login");
        browser.type("userNameInput", TestHelper.getAuthorizedStudentUsername());
        browser.type("passwordInput", TestHelper.getAuthorizedStudentPassword());
        browser.click("//div[@id='loginBoxC']/div[1]/a[1]/span/span");
        browser.waitForPageToLoad(TestHelper.getTimeout());
    	enterMyAccount();
		browser.click("link=Activity Permissions");
		browser.waitForPageToLoad(TestHelper.getTimeout());
		browser.click("link=Brando Egg Hunter");
		browser.waitForPageToLoad(TestHelper.getTimeout());
		assertTrue("permission should be correct", browser.isTextPresent("You currently do not have the permission to join this activity."));
		assertTrue("permission should be correct", browser.isTextPresent("Please request your parent's permission or try out the practice room for the time being."));
    }

    /**
     * FTC 75 2.10.1.1: Verify list of student's permissions on his/her Profile can be viewed by parent (Activities data).
     * @throws Exception if any error
     */
    public void testFTC75() throws Exception {
        browser = TestHelper.getLoggedInAsParent(browser);
    	enterMyAccount();
    	browser.click("//div[@id='noApproveTab']/div[1]/ul/li[2]/a/span/span");
		browser.waitForPageToLoad(TestHelper.getTimeout());
		browser.click("//p[@id='parentalControlSelect']/span/div/div/a");
		browser.click("link=csstemstu Ford");
		browser.waitForPageToLoad(TestHelper.getTimeout());
		assertTrue("activities must be displayed", browser.isTextPresent("Activity permissions for \"csstemstu Ford\""));
		assertTrue("activities must be displayed", browser.isElementPresent("//table[@class='generalTable generalListTable']"));
		assertTrue("activities must be displayed", browser.isElementPresent("facebookAllowed_16"));
		assertTrue("activities must be displayed", browser.isElementPresent("facebookAllowed_17"));
		assertTrue("activities must be displayed", browser.isElementPresent("facebookAllowed_18"));
		assertTrue("activities must be displayed", browser.isElementPresent("twitterAllowed_16"));
		assertTrue("activities must be displayed", browser.isElementPresent("twitterAllowed_17"));
		assertTrue("activities must be displayed", browser.isElementPresent("twitterAllowed_18"));
		assertTrue("activities must be displayed", browser.isElementPresent("allowed_16"));
		assertTrue("activities must be displayed", browser.isElementPresent("allowed_17"));
		assertTrue("activities must be displayed", browser.isElementPresent("allowed_18"));
    }

    /**
     * FTC 76 2.10.1.1: Verify list of student's permissions on his/her Profile can be viewed by parent (Forums data).
     * @throws Exception if any error
     */
    public void testFTC76() throws Exception {
        browser = TestHelper.getLoggedInAsParent(browser);
    	enterMyAccount();
    	browser.click("//div[@id='noApproveTab']/div[1]/ul/li[2]/a/span/span");
		browser.waitForPageToLoad(TestHelper.getTimeout());
		browser.click("//p[@id='parentalControlSelect']/span/div/div/a");
		browser.click("link=joebloggs");
		browser.waitForPageToLoad(TestHelper.getTimeout());
		browser.click("//div[@id='noApproveTab']/div[3]/div/div[1]/div[1]/ul/li[2]/a/span/span");
		assertTrue("forums must be displayed", browser.isTextPresent("Forum activities by"));
		assertTrue("forum header is not displayed", browser.isElementPresent("forumList_subject"));
		assertEquals("forum header is not displayed", "Subject", browser.getText("forumList_subject"));
		assertTrue("forum header is not displayed", browser.isElementPresent("forumList_body"));
		assertEquals("forum header is not displayed", "Body", browser.getText("forumList_body"));
		assertTrue("forum header is not displayed", browser.isElementPresent("forumList_id"));
		assertEquals("forum header is not displayed", "Delete", browser.getText("forumList_id"));
		assertTrue("forum topic must be displayed", browser.isElementPresent("link=exact:RE: New Thread from Romano"));
		assertTrue("forums body be displayed", browser.isTextPresent("Good boy!"));
    }

    /**
     * FTC 77 2.10.1.1: Verify list of student's permissions on his/her Profile can be viewed by parent (Blog data).
     * @throws Exception if any error
     */
    public void testFTC77() throws Exception {
        browser = TestHelper.getLoggedInAsParent(browser);
    	enterMyAccount();
    	browser.click("//div[@id='noApproveTab']/div[1]/ul/li[2]/a/span/span");
		browser.waitForPageToLoad(TestHelper.getTimeout());
		browser.click("//p[@id='parentalControlSelect']/span/div/div/a");
		browser.click("link=csstemstu Ford");
		browser.waitForPageToLoad(TestHelper.getTimeout());
		assertTrue("blogs must be displayed", browser.isTextPresent("Blog activities by \"csstemstu Ford\" :"));
    }

    /**
     * FTC 78 2.10.1.1: Verify list of student's permissions on his/her Profile can be viewed by parent (Private Messages data).
     * @throws Exception if any error
     */
    public void testFTC78() throws Exception {
        fail("no such logic");
    }

    /**
     * FTC 79 2.10.1.1: Verify pagination works well on the list of student's permissions on his/her Profile (Activities Data).
     * @throws Exception if any error
     */
    public void testFTC79() throws Exception {
        fail("no such logic");
    }

    /**
     * FTC 80 2.10.1.1: Verify pagination works well on the list of student's permissions on his/her Profile (Forums Data).
     * @throws Exception if any error
     */
    public void testFTC80() throws Exception {
        browser = TestHelper.getLoggedInAsParent(browser);
    	enterMyAccount();
    	browser.click("//div[@id='noApproveTab']/div[1]/ul/li[2]/a/span/span");
		browser.waitForPageToLoad(TestHelper.getTimeout());
		browser.click("//p[@id='parentalControlSelect']/span/div/div/a");
		browser.click("link=joebloggs");
		browser.waitForPageToLoad(TestHelper.getTimeout());
		Thread.sleep(SLEEP);
		assertTrue("forums must be displayed", browser.isTextPresent("Forum activities"));
		browser.click("//td[@id='next']/span");
		Thread.sleep(SLEEP);
		assertTrue("navigated to page 2", browser.isTextPresent("Multiple Submissions"));
		browser.click("//td[@id='prev']/span");
		Thread.sleep(SLEEP);
		assertTrue("navigated to page 1", browser.isTextPresent("New Thread from Romano"));
    }

    /**
     * FTC 81 2.10.1.1: Verify pagination works well on the list of student's permissions on his/her Profile (Blog Data).
     * @throws Exception if any error
     */
    public void testFTC81() throws Exception {
        fail("no such logic");
    }

    /**
     * FTC 81 2.10.1.1: Verify pagination works well on the list of student's permissions on his/her Profile (Private Messages Data).
     * @throws Exception if any error
     */
    public void testFTC82() throws Exception {
        fail("no such logic");
    }

    /**
     * FTC 92 2.10.1.1/2.10.1.4/2.10.1.6/2.10.1.7: Verify activities can be marked as allowed for students by parent.
     * @throws Exception if any error
     */
    public void testFTC92() throws Exception {
        browser = TestHelper.getLoggedInAsParent(browser);
    	enterMyAccount();
    	browser.click("//div[@id='noApproveTab']/div[1]/ul/li[2]/a/span/span");
		browser.waitForPageToLoad(TestHelper.getTimeout());
		browser.click("//p[@id='parentalControlSelect']/span/div/div/a");
		browser.click("link=csstemstu Ford");
		browser.waitForPageToLoad(TestHelper.getTimeout());
		assertTrue("activities must be displayed", browser.isTextPresent("Activity permissions for \"csstemstu Ford\""));
		browser.click("allowed_16");
		browser.click("//form[@id='frmPermissions']/div[4]/a[1]/span/span");
		browser.waitForPageToLoad(TestHelper.getTimeout());
		assertTrue("permission is not saved", browser.isTextPresent("Activities Permissions"));
		assertTrue("permission is not saved", browser.isTextPresent("You have successfully changed the CS-STEM activities\npermissions for your child Student."));
    }

    /**
     * FTC 93 2.10.1.1/2.10.1.4/2.10.1.6/2.10.1.7: Verify activities can be marked as allowed for students by parent.
     * @throws Exception if any error
     */
    public void testFTC93() throws Exception {
        browser = TestHelper.getLoggedInAsParent(browser);
    	enterMyAccount();
    	browser.click("//div[@id='noApproveTab']/div[1]/ul/li[2]/a/span/span");
		browser.waitForPageToLoad(TestHelper.getTimeout());
		browser.click("//p[@id='parentalControlSelect']/span/div/div/a");
		browser.click("link=csstemstu Ford");
		browser.waitForPageToLoad(TestHelper.getTimeout());
		assertTrue("activities must be displayed", browser.isTextPresent("Activity permissions for \"csstemstu Ford\""));
		browser.click("//input[@name='allowed_17' and @value='false']");
		browser.click("//form[@id='frmPermissions']/div[4]/a[1]/span/span");
		browser.waitForPageToLoad(TestHelper.getTimeout());
		assertTrue("permission is not saved", browser.isTextPresent("Activities Permissions"));
		assertTrue("permission is not saved", browser.isTextPresent("You have successfully changed the CS-STEM activities\npermissions for your child Student."));
    }

    /**
     * FTC 93 2.10.1.1/2.10.1.4/2.10.1.6/2.10.1.7: Verify permission for activities can be cancelled by parent.
     * @throws Exception if any error
     */
    public void testFTC94() throws Exception {
        browser = TestHelper.getLoggedInAsParent(browser);
    	enterMyAccount();
    	browser.click("//div[@id='noApproveTab']/div[1]/ul/li[2]/a/span/span");
		browser.waitForPageToLoad(TestHelper.getTimeout());
		browser.click("//p[@id='parentalControlSelect']/span/div/div/a");
		browser.click("link=csstemstu Ford");
		browser.waitForPageToLoad(TestHelper.getTimeout());
		assertTrue("activities must be displayed", browser.isTextPresent("Activity permissions for \"csstemstu Ford\""));
		browser.click("allowed_17");
		assertEquals("the value is wrong", "on", browser.getValue("allowed_17"));
		browser.click("//form[@id='frmPermissions']/div[4]/a[1]/span/span");
		assertTrue("permission is not cancelled", browser.isTextPresent("Activities Permissions"));
		assertTrue("permission is not cancelled", browser.isTextPresent("You have cancelled changes of CS-STEM activities\npermissions for your child Student."));
		assertEquals("the value is not restored", "on", browser.getValue("allowed_17"));
    }

    /**
     * FTC 100 2.11.1.3: Verify parent can recommend some content to child student.
     * @throws Exception if any error
     */
    public void testFTC100() throws Exception {
        browser = TestHelper.getLoggedInAsParent(browser);
    	enterMyAccount();
    	browser.click("//div[@id='noApproveTab']/div[1]/ul/li[2]/a/span/span");
		browser.waitForPageToLoad(TestHelper.getTimeout());
		browser.click("//p[@id='parentalControlSelect']/span/div/div/a");
		browser.click("link=csstemstu Ford");
		browser.waitForPageToLoad(TestHelper.getTimeout());
		fail("there is no recommend logic");
    }

    /**
     * FTC 101 2.11.1.4: Verify parent can comment content to child student.
     * @throws Exception if any error
     */
    public void testFTC101() throws Exception {
        browser = TestHelper.getLoggedInAsParent(browser);
    	enterMyAccount();
    	browser.click("//div[@id='noApproveTab']/div[1]/ul/li[2]/a/span/span");
		browser.waitForPageToLoad(TestHelper.getTimeout());
		browser.click("//p[@id='parentalControlSelect']/span/div/div/a");
		browser.click("link=csstemstu Ford");
		browser.waitForPageToLoad(TestHelper.getTimeout());
		fail("there is no comment logic");
    }

    /**
     * Get the registration page.
     * @return the page
     * @throws Exception if any error
     */
    private Selenium getRegistrationPage() throws Exception {
        browser.click("link=Login");
		browser.click("link=Register");
		browser.waitForPageToLoad(TestHelper.getTimeout());
		return browser;
	}

	/**
     * Enter my account page.
     * @throws Exception if any error
     */
    private void enterMyAccount() throws Exception {
		browser.click("//ul[@id='nav']/li[6]/a/span[2]/span");
		browser.waitForPageToLoad(TestHelper.getTimeout());
    }

    /**
     * Restore the original account.
     * @throws Exception if any error
     */
    private void restoreAccount() throws Exception {
		// restore
		browser.type("_parentaccountmanagementportlet_WAR_parentprofilepagesportlet_email", "csstem1@163.com");
		browser.click("//a[@class='greenBtn saveButton']");
		browser.waitForPageToLoad(TestHelper.getTimeout());
    }

    /**
     * Restore the original profile.
     * @throws Exception if any error
     */
    private void restoreProfile() throws Exception {
		// restore
    	browser.type("_parentmyprofileportlet_WAR_parentregistrationportlet_firstName", "first");
		browser.type("_parentmyprofileportlet_WAR_parentregistrationportlet_middleName", "middle");
		browser.type("_parentmyprofileportlet_WAR_parentregistrationportlet_lastName", "last");
		browser.type("_parentmyprofileportlet_WAR_parentregistrationportlet_country", "United States");
		browser.select("gender", "Male");
		browser.click("//a[@class='greenBtn saveButton']");
		browser.waitForPageToLoad(TestHelper.getTimeout());
    }
}
