import java.awt.*;
import java.awt.geom.*;
import java.awt.image.*;
import java.util.*;
import java.util.List;
import javax.swing.*;

public class RobonautVision {
	private int w, h, len;
	private static final boolean debug = false;
	private static final int testingLimit = 88000;
	private static final int factor = 8;
	private static final int sfactor = 3;
	private static final int borderSize = 300;
	private static final int xMin = -250, xMax = 300;
	private static final int yMin = -250, yMax = 190;
	private static final int zMin = 650, zMax = 1040;
	private static final int xMinBoxMin = -400, xMinBoxMax = 200;
	private static final int xMaxBoxMin = -220, xMaxBoxMax = 380;
	private static final int yMinBoxMin = -450, yMinBoxMax = 30;
	private static final int yMaxBoxMin = -90, yMaxBoxMax = 330;
	private static final int zMinBoxMin = 450, zMinBoxMax = 950;
	private static final int zMaxBoxMin = 750, zMaxBoxMax = 1130;
	private static final int maxModelPoints = 1500;
	private static final int minModelPointsSqDist = 10;
	private final List<Integer> rem = new ArrayList<Integer>();
	private boolean[] mask0, mask1;
	private Point3D[] modelPoints, box;
	private Face[] modelFaces;
	private String[] plyData;
	private Random rnd = new Random(2202141219080909L);

	public int trainingModel(String[] plyData) {
		try {
			this.plyData = plyData;
			len = (w = 1600) * (h = 1200);
			parseModel();
			findBox();
			interpolateModel();
			reduceModel();
		} catch (Exception ex) {
			ex.printStackTrace();
		}
		return 1;
	}

	public int trainingPair(int[] leftImage, int[] rightImage, double[] groundTruth) {
		return 1;
	}

	public int doneTraining() {
		return 1;
	}

	public double[] testingPair(int[] leftImage, int[] rightImage) {
		return testingPair(leftImage, rightImage, null);
	}

	public double[] testingPair(int[] leftImage, int[] rightImage, double[] groundTruth) {
		try {
			len = (w = 1600) * (h = 1200);
			//if (groundTruth != null) return groundTruth;
			long timeout0 = System.currentTimeMillis() + testingLimit * 1 / 10;
			long timeout1 = System.currentTimeMillis() + testingLimit * 8 / 10;
			long timeout2 = System.currentTimeMillis() + testingLimit * 9 / 10;
			long timeout3 = System.currentTimeMillis() + testingLimit;

			rnd = new Random(197209091220L);
			despeckle(leftImage);
			despeckle(rightImage);
			stretch(leftImage);
			stretch(rightImage);
			leftImage = resize(leftImage);
			rightImage = resize(rightImage);
			len = (w /= factor) * (h /= factor);
			mask0 = new boolean[len];
			mask1 = new boolean[len];
			int border = borderSize / factor;
			/*if (debug) {
				BufferedImage img1 = new BufferedImage(w, h, BufferedImage.TYPE_INT_RGB);
				for (int y = 0; y < h; y++) {
					for (int x = 0; x < w; x++) {
						img1.setRGB(x, y, leftImage[y * w + x]);
					}
				}
				new ImgViewer(img1);
			}*/

			Rectangle2D rcAll = new Rectangle2D.Double(0, 0, w, h);
			Area hotArea = new Area(rcAll);
			double minArea = w * h * 0.04;
			int skip1 = 0;

			List<Point2D> tBox0 = new ArrayList<Point2D>();
			List<Point2D> tBox1 = new ArrayList<Point2D>();

			int[] leftDif = difference(leftImage, rcAll);
			int[] rightDif = difference(rightImage, rcAll);

			long t = System.currentTimeMillis();
			Beam<Position> bestPos = new Beam<Position>(5);
			for (int i = 0; i < 100000; i++) {
				long time = System.currentTimeMillis();
				if (time > timeout0) break;
				double[] npos = randomPos();
				tBox0.clear();
				tBox1.clear();
				if (!checkBox(tBox0, tBox1, npos)) {
					skip1++;
					continue;
				}
				if (intersectArea(hotArea, tBox0) + intersectArea(hotArea, tBox1) < minArea) {
					skip1++;
					continue;
				}
				double curr = eval(leftImage, rightImage, leftDif, rightDif, npos, rcAll, rcAll, false, false);
				bestPos.add(new Position(curr, npos));
			}
			if (debug) System.err.println("SKIP0=" + skip1);

			Rectangle2D[] rc = new Rectangle2D[2];
			for (int side = 0; side <= 1; side++) {
				rc[side] = new Rectangle2D.Double(0, 0, w, h);
				for (int i = 0; i < bestPos.size(); i++) {
					double[] pos = bestPos.get(i).pos;
					int x0 = 0;
					int x1 = 0;
					int y0 = 0;
					int y1 = 0;
					for (int j = 0; j < modelPoints.length; j++) {
						Point2D[] p = Transform.transform3Dto2D(Transform.applyPos(modelPoints[j], pos));
						Point2D pk = p[side];
						int px = (int) Math.round(pk.x / factor);
						int py = (int) Math.round(pk.y / factor);
						if (j == 0) {
							x0 = x1 = px;
							y0 = y1 = py;
						}
						if (x0 > px) x0 = px;
						if (y0 > py) y0 = py;
						if (x1 < px) x1 = px;
						if (y1 < py) y1 = py;

					}
					Rectangle2D ri = new Rectangle2D.Double(x0, y0, x1 - x0 + 1, y1 - y0 + 1);
					Rectangle2D rn = rc[side].createIntersection(ri);
					if (rn.getWidth() <= 0 || rn.getHeight() <= 0) break;
					rc[side] = rn;
				}
			}
			t = System.currentTimeMillis() - t;
			if (debug) System.err.println("FindRect : " + t + " ms");

			t = System.currentTimeMillis();
			Rectangle2D[] rcOrg = new Rectangle2D[2];
			rcOrg[0] = rc[0].createUnion(rc[0]);
			rcOrg[1] = rc[1].createUnion(rc[1]);
			refineRect(rc[0], leftDif);
			refineRect(rc[1], rightDif);
			t = System.currentTimeMillis() - t;
			if (debug) System.err.println("RefineRect : " + t + " ms");

			if (debug) {
				for (int side = 0; side <= 1; side++) {
					int[] a = side == 0 ? leftDif : rightDif;
					BufferedImage img = new BufferedImage(w, h, BufferedImage.TYPE_INT_RGB);
					for (int y = 0; y < h; y++) {
						for (int x = 0; x < w; x++) {
							int c = a[y * w + x];
							img.setRGB(x, y, c * 65793);
						}
					}
					Graphics2D g = (Graphics2D) img.getGraphics();
					g.setStroke(new BasicStroke(1f));
					g.setColor(Color.green);
					g.draw(rcOrg[side]);
					g.setColor(Color.yellow);
					g.draw(rc[side]);
					g.dispose();
					new ImgViewer(img);
				}
			}
			t = System.currentTimeMillis();

			Area hotArea0 = new Area(rc[0]);
			Area hotArea1 = new Area(rc[1]);
			double minArea0 = rc[0].getWidth() * rc[0].getHeight() * 0.9;
			double minArea1 = rc[1].getWidth() * rc[1].getHeight() * 0.9;
			skip1 = 0;

			Rectangle2D rcEval0 = new Rectangle2D.Double(rc[0].getX() - border, rc[0].getY() - border, rc[0].getWidth() + 2 * border, rc[0].getHeight() + 2 * border).createIntersection(rcAll);
			Rectangle2D rcEval1 = new Rectangle2D.Double(rc[1].getX() - border, rc[1].getY() - border, rc[1].getWidth() + 2 * border, rc[1].getHeight() + 2 * border).createIntersection(rcAll);

			leftDif = difference(leftImage, rcEval0);
			rightDif = difference(rightImage, rcEval1);

			double[] pos = bestPos.get(0).pos;
			double val = eval(leftImage, rightImage, leftDif, rightDif, pos, rcEval0, rcEval1, true, false);
			bestPos.clear();
			bestPos.setWidth(1);
			bestPos.add(new Position(val, pos));

			int skip2 = 0;
			int e2 = 0;
			int mode = 0;
			for (int i = 0;; i++) {
				long time = System.currentTimeMillis();
				if (time > timeout3) break;
				if (mode == 0 && time > timeout1) {
					mode = 1;
					rnd = new Random(20120622L);
				}
				if (mode == 1 && time > timeout2) {
					mode = 2;
					rnd = new Random(20140819L);
					pos = bestPos.get(0).pos;
					val = eval(leftImage, rightImage, leftDif, rightDif, pos, rcEval0, rcEval1, true, false);
					bestPos.clear();
					bestPos.add(new Position(val, pos));
					if (debug) System.err.println("---->" + val);
				}
				double[] npos = mode >= 1 ? randomPos(bestPos.get(0).pos) : randomPos();
				//npos[0] = groundTruth[0];
				//npos[1] = groundTruth[1];
				//npos[2] = groundTruth[2];

				tBox0.clear();
				tBox1.clear();
				if (!checkBox(tBox0, tBox1, npos)) {
					skip2++;
					continue;
				}
				if (intersectArea(hotArea0, tBox0) < minArea0) {
					skip1++;
					continue;
				}
				if (intersectArea(hotArea1, tBox1) < minArea1) {
					skip1++;
					continue;
				}
				e2++;

				double curr = eval(leftImage, rightImage, leftDif, rightDif, npos, rcEval0, rcEval1, mode == 2, false);
				if (bestPos.add(new Position(curr, npos))) {
					if (debug) System.err.println(i + " >> " + curr);
				}
			}
			if (debug) {
				System.err.println("SKIP1=" + skip1);
				System.err.println("SKIP2=" + skip2);
				System.err.println("EVAL2=" + e2);
				System.err.println();
			}
			pos = bestPos.get(0).pos;
			eval(leftImage, rightImage, leftDif, rightDif, pos, rcEval0, rcEval1, true, true);
			if (groundTruth != null) {
				eval(leftImage, rightImage, leftDif, rightDif, groundTruth, rcEval0, rcEval1, true, true);
			}
			t = System.currentTimeMillis() - t;
			if (debug) {
				System.err.println();
				System.err.println("Evals : " + t + " ms");
			}
			return Arrays.copyOf(pos, 7);
		} catch (Exception ex) {
			System.err.println(">>>>>>>>>>>>>Exception: " + ex.getMessage());
			ex.printStackTrace();
			return randomPos();
		}
	}

	private double intersectArea(Area hotArea, List<Point2D> l) throws Exception {
		List<Point2D> c1 = Geom.convexHull(l);
		GeneralPath gp = new GeneralPath();
		gp.moveTo(c1.get(0).x, c1.get(0).y);
		for (int j = 1; j < c1.size(); j++) {
			gp.lineTo(c1.get(j).x, c1.get(j).y);
		}
		gp.closePath();
		Area a = new Area(gp);
		a.intersect(hotArea);
		Rectangle2D ra = a.getBounds2D();
		if (ra.getWidth() < 0 || ra.getHeight() < 0) return 0;
		return ra.getWidth() * ra.getHeight();
	}

	private boolean checkBox(List<Point2D> tBox0, List<Point2D> tBox1, double[] pos) throws Exception {
		double minX = 1e10;
		double maxX = -1e10;
		double minY = 1e10;
		double maxY = -1e10;
		double minZ = 1e10;
		double maxZ = -1e10;
		for (int j = 0; j < box.length; j++) {
			Point3D pt = Transform.applyPos(box[j], pos);
			if (minX > pt.x) minX = pt.x;
			if (maxX < pt.x) maxX = pt.x;
			if (minY > pt.y) minY = pt.y;
			if (maxY < pt.y) maxY = pt.y;
			if (minZ > pt.z) minZ = pt.z;
			if (maxZ < pt.z) maxZ = pt.z;
			Point2D[] p = Transform.transform3Dto2D(pt);
			tBox0.add(p[0].resize(factor));
			tBox1.add(p[1].resize(factor));
		}
		if (minX < xMinBoxMin || minX > xMinBoxMax) return false;
		if (maxX < xMaxBoxMin || maxX > xMaxBoxMax) return false;
		if (minY < yMinBoxMin || minY > yMinBoxMax) return false;
		if (maxY < yMaxBoxMin || maxY > yMaxBoxMax) return false;
		if (minZ < zMinBoxMin || minZ > zMinBoxMax) return false;
		if (maxZ < zMaxBoxMin || maxZ > zMaxBoxMax) return false;
		return true;
	}

	private void refineRect(Rectangle2D rc, int[] difference) throws Exception {
		int x0 = (int) rc.getX();
		int x1 = x0 + (int) rc.getWidth();
		int y0 = (int) rc.getY();
		int y1 = y0 + (int) rc.getHeight();
		if (x0 <= 0) x0 = 1;
		if (y0 <= 0) y0 = 1;
		if (x1 >= w - 1) x1 = w - 2;
		if (y1 >= h - 1) y1 = h - 2;
		final double Q = 0.3;
		final double R1 = 20;
		final double R2 = 5;
		final int D = 50;
		for (int i = 0; i < 2000; i++) {
			int mode = rnd.nextInt(8);
			if (mode == 0 && x0 > 1) {
				double a = evalYLine(x0, y0, y1, difference);
				double b = evalYLine(x0 - 1, y0, y1, difference);
				if (b > R1 && b != 0 && Math.abs(a / b - 1) < Q) x0--;
			}
			if (mode == 1 && x1 < w - 2) {
				double a = evalYLine(x1, y0, y1, difference);
				double b = evalYLine(x1 + 1, y0, y1, difference);
				if (b > R1 && b != 0 && Math.abs(a / b - 1) < Q) x1++;
			}
			if (mode == 2 && x0 < x1 - D) {
				double a = evalYLine(x0, y0, y1, difference);
				double b = evalYLine(x0 + 1, y0, y1, difference);
				if (b < R2 && b != 0 && Math.abs(a / b - 1) < Q) x0++;
			}
			if (mode == 3 && x1 > x0 + D) {
				double a = evalYLine(x1, y0, y1, difference);
				double b = evalYLine(x1 - 1, y0, y1, difference);
				if (b < R2 && b != 0 && Math.abs(a / b - 1) < Q) x1--;
			}
			if (mode == 4 && y0 > 1) {
				double a = evalXLine(y0, x0, x1, difference);
				double b = evalXLine(y0 - 1, x0, x1, difference);
				if (b > R1 && b != 0 && Math.abs(a / b - 1) < Q) y0--;
			}
			if (mode == 5 && y1 < h - 2) {
				double a = evalXLine(y1, x0, x1, difference);
				double b = evalXLine(y1 + 1, x0, x1, difference);
				if (b > R1 && b != 0 && Math.abs(a / b - 1) < Q) y1++;
			}
			if (mode == 6 && y0 < y1 - D) {
				double a = evalXLine(y0, x0, x1, difference);
				double b = evalXLine(y0 + 1, x0, x1, difference);
				if (b < R2 && b != 0 && Math.abs(a / b - 1) < Q) y0++;
			}
			if (mode == 7 && y1 > y0 + D) {
				double a = evalXLine(y1, x0, x1, difference);
				double b = evalXLine(y1 - 1, x0, x1, difference);
				if (b < R2 && b != 0 && Math.abs(a / b - 1) < Q) y1--;
			}
		}
		rc.setRect(x0, y0, x1 - x0, y1 - y0);
	}

	private double evalYLine(int x, int y0, int y1, int[] difference) throws Exception {
		double v = 0;
		for (int y = y0; y < y1; y++) {
			v += difference[y * w + x];
			v += difference[y * w + x + 1];
			v += difference[y * w + x - 1];
		}
		return v / (y1 - y0) / 3;
	}

	private double evalXLine(int y, int x0, int x1, int[] difference) throws Exception {
		double v = 0;
		for (int x = x0; x < x1; x++) {
			v += difference[y * w + x];
			v += difference[y * w + x + w];
			v += difference[y * w + x - w];
		}
		return v / (x1 - x0) / 3;
	}

	private double[] randomPos() {
		double[] npos = new double[10];
		for (int j = 0; j < 6; j++) {
			if (j < 3) {
				int min = j == 0 ? xMin : j == 1 ? yMin : zMin;
				int max = j == 0 ? xMax : j == 1 ? yMax : zMax;
				npos[j] = rnd.nextInt(max - min + 1) + min;
			} else {
				npos[j + 4] = 2 * Math.PI * (rnd.nextDouble() - 0.5);
			}
		}
		double[] q = toQuaternion(npos[7], npos[8], npos[9]);
		System.arraycopy(q, 0, npos, 3, 4);
		return npos;
	}

	private double[] randomPos(double[] pos) throws Exception {
		double[] npos = new double[10];
		for (int j = 0; j < 6; j++) {
			if (j < 3) {
				npos[j] = pos[j] + rnd.nextInt(41) - 20;
			} else {
				npos[j + 4] = Math.PI * (rnd.nextDouble() - 0.5) / 10 + pos[j + 4];
			}
		}
		double[] q = toQuaternion(npos[7], npos[8], npos[9]);
		System.arraycopy(q, 0, npos, 3, 4);
		return npos;
	}

	private void despeckle(int[] img) throws Exception {
		long t = System.currentTimeMillis();
		for (int y = 0; y < h; y++) {
			int idx = y * w;
			for (int x = 0; x < w; x++, idx++) {
				int p = img[idx];
				int r = r(p);
				int g = g(p);
				int b = b(p);
				if ((r != 255 && r != 0) || (g != 255 && g != 0) || (b != 255 && b != 0)) continue;
				int cnt = 0;
				int ar = 0;
				int ag = 0;
				int ab = 0;
				for (int dy = -1; dy <= 1; dy++) {
					if (dy + y < 0 || dy + y >= h) continue;
					for (int dx = -1; dx <= 1; dx++) {
						if (dx + x < 0 || dx + x >= w) continue;
						int q = img[idx + dy * w + dx];
						int r2 = r(q);
						int g2 = g(q);
						int b2 = b(q);
						if ((r2 == 255 || r2 == 0 || g2 == 255 || g2 == 0 || b2 == 255 && b2 == 0)) continue;
						ar += r2;
						ag += g2;
						ab += b2;
						cnt++;
					}
				}
				if (cnt > 0) img[idx] = rgb(ar / cnt, ag / cnt, ab / cnt);
			}
		}
		t = System.currentTimeMillis() - t;
		if (debug) System.err.println("Despeckle : " + t + " ms");
	}

	private void stretch(int[] img) throws Exception {
		if (debug) {
			BufferedImage v = new BufferedImage(w, h, BufferedImage.TYPE_INT_RGB);
			for (int y = 0; y < h; y++) {
				for (int x = 0; x < w; x++) {
					int c = img[y * w + x];
					v.setRGB(x, y, c);
				}
			}
			new ImgViewer(v);
		}
		long t = System.currentTimeMillis();
		int min = 255;
		int max = 0;
		for (int y = 0; y < h; y++) {
			int idx = y * w;
			for (int x = 0; x < w; x++, idx++) {
				int p = img[idx];
				int r = r(p);
				int g = g(p);
				int b = b(p);
				if (min > r) min = r;
				if (min > g) min = g;
				if (min > b) min = b;
				if (max < r) max = r;
				if (max < g) max = g;
				if (max < b) max = b;
			}
		}
		if (debug) System.err.println("HIST.RANGE = " + min + " : " + max);
		if (min > 0 || max < 255) {
			for (int y = 0; y < h; y++) {
				int idx = y * w;
				for (int x = 0; x < w; x++, idx++) {
					int p = img[idx];
					int r = (r(p) - min) * 255 / (max - min);
					if (r > 255) r = 255;
					int g = (g(p) - min) * 255 / (max - min);
					if (g > 255) g = 255;
					int b = (b(p) - min) * 255 / (max - min);
					if (b > 255) b = 255;
					img[idx] = rgb(r, g, b);
				}
			}
		}
		t = System.currentTimeMillis() - t;
		if (debug) System.err.println("Stretch : " + t + " ms");
		if (debug) {
			BufferedImage v = new BufferedImage(w, h, BufferedImage.TYPE_INT_RGB);
			for (int y = 0; y < h; y++) {
				for (int x = 0; x < w; x++) {
					int c = img[y * w + x];
					v.setRGB(x, y, c);
				}
			}
			new ImgViewer(v);
		}
	}

	private int[] resize(int[] img) throws Exception {
		int[] ret = new int[(w / factor) * (h / factor)];
		for (int y = 0; y < h; y += factor) {
			for (int x = 0; x < w; x += factor) {
				int ar = 0;
				int ag = 0;
				int ab = 0;
				int cnt = 0;
				for (int dy = 0; dy < factor; dy++) {
					for (int dx = 0; dx < factor; dx++) {
						cnt++;
						int p = img[(y + dy) * w + (x + dx)];
						ar += r(p);
						ag += g(p);
						ab += b(p);
						cnt++;
					}
				}
				ret[(y / factor) * (w / factor) + (x / factor)] = (ar / cnt << 16) | (ag / cnt << 8) | (ab / cnt);
			}
		}
		return ret;
	}

	private static final int rgb(int r, int g, int b) throws Exception {
		return (r << 16) | (g << 8) | b;
	}

	private static final int r(int rgb) throws Exception {
		return (rgb >>> 16) & 255;
	}

	private static final int g(int rgb) throws Exception {
		return (rgb >>> 8) & 255;
	}

	private static final int b(int rgb) throws Exception {
		return rgb & 255;
	}

	private double eval(int[] leftImg, int[] rightImg, int[] leftDif, int[] rightDif, double[] pos, Rectangle2D rc0, Rectangle2D rc1, boolean refine, boolean show) throws Exception {
		Arrays.fill(mask0, false);
		Arrays.fill(mask1, false);
		final int k = 8 / factor;
		for (int i = 0; i < modelPoints.length; i++) {
			Point3D pt = Transform.applyPos(modelPoints[i], pos);
			Point2D[] p = Transform.transform3Dto2D(pt);
			Point2D pj0 = p[0];
			Point2D pj1 = p[1];

			int px0 = ((int) Math.round(pj0.x)) >> sfactor;
			int py0 = ((int) Math.round(pj0.y)) >> sfactor;

			int px1 = ((int) Math.round(pj1.x)) >> sfactor;
			int py1 = ((int) Math.round(pj1.y)) >> sfactor;

			for (int dy = -k; dy <= k; dy++) {
				int i0 = (py0 + dy) * w + px0 - k;
				int i1 = (py1 + dy) * w + px1 - k;
				for (int dx = -k; dx <= k; dx++, i0++, i1++) {
					if (px0 + dx >= 0 && px0 + dx < w && py0 + dy >= 0 && py0 + dy < h) mask0[i0] = true;
					if (px1 + dx >= 0 && px1 + dx < w && py1 + dy >= 0 && py1 + dy < h) mask1[i1] = true;
				}
			}
		}
		double[] v0 = eval(mask0, rc0, leftDif, k, refine);
		double[] v1 = eval(mask1, rc1, rightDif, k, refine);
		double val = (v0[1] + v1[1] + Math.min(v0[1], v1[1])) + 10 * (v0[0] + v1[0] + Math.min(v0[0], v1[0]));
		if (show && debug) {
			BufferedImage img0 = new BufferedImage(w, h, BufferedImage.TYPE_INT_RGB);
			BufferedImage img1 = new BufferedImage(w, h, BufferedImage.TYPE_INT_RGB);
			for (int y = 0; y < h; y++) {
				for (int x = 0; x < w; x++) {
					int idx = y * w + x;

					int c0 = leftDif[idx] * 65793;
					if (mask0[idx]) c0 = 255;
					img0.setRGB(x, y, c0);

					int c1 = rightDif[idx] * 65793;
					if (mask1[idx]) c1 = 255;
					img1.setRGB(x, y, c1);
				}
			}
			Graphics2D g0 = (Graphics2D) img0.getGraphics();
			g0.setRenderingHint(RenderingHints.KEY_ANTIALIASING, RenderingHints.VALUE_ANTIALIAS_ON);
			g0.setStroke(new BasicStroke(0.5f));
			g0.setColor(Color.yellow);
			g0.dispose();

			new ImgViewer(img0);
			new ImgViewer(img1);
			System.err.println(val + "\t" + (v0[0] + v1[0]) + "\t" + (v0[1] + v1[1]));
		}
		return val;
	}

	private double[] eval(boolean[] mask, Rectangle2D rc, int[] dif, int k, boolean refine) throws Exception {
		int kd = 4 * k;
		int y0 = (int) rc.getY();
		int x0 = (int) rc.getX();
		int y1 = (int) (rc.getY() + rc.getHeight());
		int x1 = (int) (rc.getX() + rc.getWidth());
		if (refine) {
			for (int i = 0; i < k; i++) {
				rem.clear();
				for (int y = y0; y < y1; y++) {
					int idx = y * w + x0;
					int c = 0;
					for (int x = x0; x < x1; x++, idx++) {
						if (mask[idx]) {
							if (c > kd) rem.add(idx);
							c = 0;
						} else c++;
					}
					c = 0;
					idx = y * w + x1 - 1;
					for (int x = x1 - 1; x >= x0; x--, idx--) {
						if (mask[idx]) {
							if (c > kd) rem.add(idx);
							c = 0;
						} else c++;
					}
				}
				for (int x = x0; x < x1; x++) {
					int c = 0;
					for (int y = y0; y < y1; y++) {
						int idx = y * w + x;
						if (mask[idx]) {
							if (c > kd) rem.add(idx);
							c = 0;
						} else c++;
					}
					c = 0;
					for (int y = y1 - 1; y >= y0; y--) {
						int idx = y * w + x;
						if (mask[idx]) {
							if (c > kd) rem.add(idx);
							c = 0;
						} else c++;
					}
				}

				for (int j = 0; j < rem.size(); j++) {
					mask[rem.get(j)] = false;
				}
			}
		}
		double[] v = new double[2];
		for (int y = y0; y < y1; y++) {
			int idx = y * w + x0;
			for (int x = x0; x < x1; x++, idx++) {
				boolean m = mask[idx];
				v[m ? 1 : 0] -= sq(m ? 255 - dif[idx] : dif[idx]);
			}
		}
		return v;
	}

	private static final int sq(int x) {
		return x * x;
	}

	private static final double sq(double x) {
		return x * x;
	}

	private final double[] toQuaternion(double ax, double ay, double az) {
		double sx = Math.sin(ax / 2);
		double sy = Math.sin(ay / 2);
		double sz = Math.sin(az / 2);
		double cx = Math.cos(ax / 2);
		double cy = Math.cos(ay / 2);
		double cz = Math.cos(az / 2);
		double cycz = cy * cz;
		double sysz = sy * sz;
		double d = cycz * cx - sysz * sx;
		double a = cycz * sx + sysz * cx;
		double b = sy * cz * cx + cy * sz * sx;
		double c = cy * sz * cx - sy * cz * sx;
		double m = Math.sqrt(sq(a) + sq(b) + sq(c) + sq(d));
		return new double[] {a / m,b / m,c / m,d / m};
	}

	private int[] difference(int[] img, Rectangle2D rc) throws Exception {
		long t = System.currentTimeMillis();
		int[] ret = new int[len];
		List<Integer> r = new ArrayList<Integer>();
		List<Integer> g = new ArrayList<Integer>();
		List<Integer> b = new ArrayList<Integer>();
		int windowSize = 1;
		for (int y = 0; y < h; y++) {
			r.clear();
			g.clear();
			b.clear();
			for (int x = 0; x < w; x++) {
				for (int i = -windowSize; i <= windowSize; i++) {
					if (i + y >= 0 && i + y < h) {
						int p = img[(y + i) * w + x];
						r.add(r(p));
						g.add(g(p));
						b.add(b(p));
					}
				}
			}
			Collections.sort(r);
			Collections.sort(g);
			Collections.sort(b);
			int rm = r.get(r.size() >>> 1);
			int gm = g.get(g.size() >>> 1);
			int bm = b.get(b.size() >>> 1);
			for (int x = 0; x < w; x++) {
				int idx = y * w + x;
				int p = img[idx];
				ret[idx] = sq(r(p) - rm) + sq(g(p) - gm) + sq(b(p) - bm);
			}
		}
		for (int x = 0; x < w; x++) {
			r.clear();
			g.clear();
			b.clear();
			for (int y = 0; y < h; y++) {
				for (int i = -windowSize; i <= windowSize; i++) {
					if (i + x >= 0 && i + x < w) {
						int p = img[y * w + x + i];
						r.add(r(p));
						g.add(g(p));
						b.add(b(p));
					}
				}
			}
			Collections.sort(r);
			Collections.sort(g);
			Collections.sort(b);
			int rm = r.get(r.size() >>> 1);
			int gm = g.get(g.size() >>> 1);
			int bm = b.get(b.size() >>> 1);
			for (int y = 0; y < h; y++) {
				int idx = y * w + x;
				int p = img[idx];
				ret[idx] = Math.min(ret[idx], sq(r(p) - rm) + sq(g(p) - gm) + sq(b(p) - bm));
			}
		}

		int[] vals = new int[ret.length];
		for (int dy = 0; dy < rc.getHeight(); dy++) {
			int y = (int) (rc.getY() + dy);
			if (y < 0 || y >= h) continue;
			for (int dx = 0; dx < rc.getWidth(); dx++) {
				int x = (int) (rc.getX() + dx);
				if (x < 0 || x >= w) continue;
				int idx = y * w + x;
				vals[idx] = ret[idx];
			}
		}
		Arrays.sort(vals);
		int max = vals[vals.length * 99 / 100];

		for (int i = 0; i < ret.length; i++) {
			if (ret[i] > max) ret[i] = max;
			ret[i] = (int) (ret[i] * 255L / max);
		}

		t = System.currentTimeMillis() - t;
		//System.err.println("Difference : " + t + " ms");

		if (debug) {
			BufferedImage v = new BufferedImage(w, h, BufferedImage.TYPE_INT_RGB);
			for (int y = 0; y < h; y++) {
				for (int x = 0; x < w; x++) {
					int c = ret[y * w + x];
					v.setRGB(x, y, c * 65793);
				}
			}
			new ImgViewer(v);
		}

		return ret;
	}

	private void interpolateModel() throws Exception {
		long t = System.currentTimeMillis();
		List<Point3D> pts = new ArrayList<Point3D>();
		for (int i = 1; i < modelPoints.length; i++) {
			pts.add(modelPoints[i]);
		}
		for (int i = 0; i < modelFaces.length; i++) {
			Face f = modelFaces[i];
			if (f.pts.length != 3) continue;
			Point3D p1 = f.pts[0];
			Point3D p2 = f.pts[1];
			Point3D p3 = f.pts[2];
			int d = (int) (0.02 * Math.sqrt(sq(p2.x * p3.y - p3.x * p2.y) + sq(p3.x * p1.y - p1.x * p3.y) + (p1.x * p2.y - p2.x * p1.y)));
			for (int k = 0; k < d; k++) {
				Point3D g = new Point3D(0, 0, 0);
				double s = 0;
				for (int j = 0; j < f.pts.length; j++) {
					double wj = rnd.nextDouble() + 0.01;
					Point3D p = f.pts[j];
					g.x += p.x * wj;
					g.y += p.y * wj;
					g.z += p.z * wj;
					s += wj;
				}
				g.x /= s;
				g.y /= s;
				g.z /= s;
				pts.add(g);
			}
		}
		modelPoints = pts.toArray(new Point3D[0]);
		t = System.currentTimeMillis() - t;
		System.err.println("InterpolateModel : " + t + " ms");
	}

	private void reduceModel() throws Exception {
		long t = System.currentTimeMillis();
		List<Point3D> l = new ArrayList<Point3D>(maxModelPoints);
		Point3D initial = modelPoints[0];
		l.add(initial);
		List<Point3D> pts = new ArrayList<Point3D>();
		for (int i = 0; i < modelPoints.length; i++) {
			pts.add(modelPoints[i]);
		}
		List<Double> minDist = new ArrayList<Double>();
		for (int i = 0; i < pts.size(); i++) {
			minDist.add(pts.get(i).sqDist(initial));
		}
		while (l.size() < maxModelPoints && !pts.isEmpty()) {
			double far = -1;
			int idx = -1;
			for (int i = 0; i < pts.size(); i++) {
				double close = minDist.get(i);
				if (close > far) {
					far = close;
					idx = i;
				}
			}
			if (idx < 0) break;
			Point3D add = pts.get(idx);
			Collections.swap(pts, idx, pts.size() - 1);
			Collections.swap(minDist, idx, pts.size() - 1);
			pts.remove(pts.size() - 1);
			minDist.remove(minDist.size() - 1);
			l.add(add);
			for (int i = 0; i < pts.size(); i++) {
				double cd = minDist.get(i);
				double nd = pts.get(i).sqDist(add);
				if (nd < cd) {
					if (nd < minModelPointsSqDist) {
						Collections.swap(pts, i, pts.size() - 1);
						Collections.swap(minDist, i, pts.size() - 1);
						pts.remove(pts.size() - 1);
						minDist.remove(minDist.size() - 1);
						i--;
					} else {
						minDist.set(i, nd);
					}
				}
			}
		}
		modelPoints = l.toArray(new Point3D[0]);
		t = System.currentTimeMillis() - t;
		System.err.println("ReduceModel : " + t + " ms");
	}

	private void parseModel() throws Exception {
		long t = System.currentTimeMillis();
		int numVertex = 0;
		int numFaces = 0;
		int idx = 0;
		while (!plyData[idx].equals("end_header")) {
			String[] tok = plyData[idx].split(" ");
			if (tok[0].equals("element")) {
				if (tok[1].equals("vertex")) {
					numVertex = Integer.parseInt(tok[2]);
				} else if (tok[1].equals("face")) {
					numFaces = Integer.parseInt(tok[2]);
				}
			}
			idx++;
		}
		idx++;
		modelPoints = new Point3D[numVertex];
		for (int i = 0; i < numVertex; i++) {
			String[] tok = plyData[idx++].split(" ");
			modelPoints[i] = new Point3D(Double.parseDouble(tok[0]), Double.parseDouble(tok[1]), Double.parseDouble(tok[2]));
		}
		modelFaces = new Face[numFaces];
		for (int i = 0; i < numFaces; i++, idx++) {
			if (idx >= plyData.length) {
				modelFaces[i] = modelFaces[i - 1];
			} else {
				String[] tok = plyData[idx].split(" ");
				Point3D[] vp = new Point3D[Integer.parseInt(tok[0])];
				int[] vi = new int[vp.length];
				for (int j = 0; j < vp.length; j++) {
					vp[j] = modelPoints[vi[j] = Integer.parseInt(tok[1 + j])];
				}
				modelFaces[i] = new Face(vp, vi);
			}
		}
		t = System.currentTimeMillis() - t;
		System.err.println("ParseModel : " + t + " ms");
	}

	private void findBox() throws Exception {
		long t = System.currentTimeMillis();
		double xMin = modelPoints[0].x;
		double xMax = modelPoints[0].x;
		double yMin = modelPoints[0].y;
		double yMax = modelPoints[0].y;
		double zMin = modelPoints[0].z;
		double zMax = modelPoints[0].z;
		for (int i = 1; i < modelPoints.length; i++) {
			if (xMin > modelPoints[i].x) xMin = modelPoints[i].x;
			if (xMax < modelPoints[i].x) xMax = modelPoints[i].x;
			if (yMin > modelPoints[i].y) yMin = modelPoints[i].y;
			if (yMax < modelPoints[i].y) yMax = modelPoints[i].y;
			if (zMin > modelPoints[i].z) zMin = modelPoints[i].z;
			if (zMax < modelPoints[i].z) zMax = modelPoints[i].z;
		}
		box = new Point3D[8];
		box[0] = new Point3D(xMin, yMin, zMin);
		box[1] = new Point3D(xMin, yMax, zMin);
		box[2] = new Point3D(xMax, yMax, zMin);
		box[3] = new Point3D(xMax, yMin, zMin);
		box[4] = new Point3D(xMin, yMin, zMax);
		box[5] = new Point3D(xMin, yMax, zMax);
		box[6] = new Point3D(xMax, yMax, zMax);
		box[7] = new Point3D(xMax, yMin, zMax);
		t = System.currentTimeMillis() - t;
		System.err.println("FindBox : " + t + " ms");
	}
}

class Point3D extends Point2D {
	double z;

	public Point3D(double x, double y, double z) {
		super(x, y);
		this.z = z;
	}

	public double sqDist(Point3D p) {
		return sq(p.x - x) + sq(p.y - y) + sq(p.z - z);
	}

	private static final double sq(double x) {
		return x * x;
	}
}

class Point2D {
	double x, y;

	public Point2D(double x, double y) {
		this.x = x;
		this.y = y;
	}

	public Point2D resize(int factor) {
		x /= factor;
		y /= factor;
		return this;
	}

	public boolean equals(Object obj) {
		return sqDist((Point2D) obj) < 1e-6;
	}

	public double sqDist(Point2D p) {
		return sq(p.x - x) + sq(p.y - y);
	}

	private static final double sq(double x) {
		return x * x;
	}
}

class Face {
	Point3D[] pts;
	int[] idx;

	public Face(Point3D[] pts, int[] idx) {
		this.pts = pts;
		this.idx = idx;
	}
}

class Transform {
	private static final double[] T = new double[] {145.626654332161,1.65379695634088,-3.65966860066967};
	private static final double[] Q = new double[] {0.999985,0.0000680463,-7.05536E-7,0.00540286};
	private static final double CxL = 0.000529185;
	private static final double CyL = 0.000528196;
	private static final double CxR = 0.000529527;
	private static final double CyR = 0.000528341;
	private static final double KL = -2.41479E-8;
	private static final double KR = -2.21532E-8;
	private static final double[] C0 = new double[] {800,600};

	private static final double[] crossProduct(double[] lhs, double[] rhs) {
		return new double[] {lhs[1] * rhs[2] - lhs[2] * rhs[1],lhs[2] * rhs[0] - lhs[0] * rhs[2],lhs[0] * rhs[1] - lhs[1] * rhs[0]};
	}

	public static final double[] rotate(double x, double y, double z, double qr, double qi, double qj, double qk) {
		double[] v = new double[] {x,y,z};
		double[] qvec = new double[] {qi,qj,qk};
		double[] uv = crossProduct(qvec, v);
		double[] uuv = crossProduct(qvec, uv);
		qr *= 2;
		uv[0] *= qr;
		uv[1] *= qr;
		uv[2] *= qr;
		uuv[0] *= 2;
		uuv[1] *= 2;
		uuv[2] *= 2;
		return new double[] {v[0] + uv[0] + uuv[0],v[1] + uv[1] + uuv[1],v[2] + uv[2] + uuv[2]};
	}

	public static final Point3D rotate(Point3D p, double[] q) {
		double[] v = new double[] {p.x,p.y,p.z};
		double[] qvec = new double[] {q[1],q[2],q[3]};
		double[] uv = crossProduct(qvec, v);
		double[] uuv = crossProduct(qvec, uv);
		double qr = 2 * q[0];
		uv[0] *= qr;
		uv[1] *= qr;
		uv[2] *= qr;
		uuv[0] *= 2;
		uuv[1] *= 2;
		uuv[2] *= 2;
		return new Point3D(v[0] + uv[0] + uuv[0], v[1] + uv[1] + uuv[1], v[2] + uv[2] + uuv[2]);
	}

	public static final Point3D applyPos(Point3D p, double[] pos) {
		double[] v = new double[] {p.x,p.y,p.z};
		double[] qvec = new double[] {pos[4],pos[5],pos[6]};
		double[] uv = crossProduct(qvec, v);
		double[] uuv = crossProduct(qvec, uv);
		double qr = 2 * pos[3];
		uv[0] *= qr;
		uv[1] *= qr;
		uv[2] *= qr;
		uuv[0] *= 2;
		uuv[1] *= 2;
		uuv[2] *= 2;
		return new Point3D(v[0] + uv[0] + uuv[0] + pos[0], v[1] + uv[1] + uuv[1] + pos[1], v[2] + uv[2] + uuv[2] + pos[2]);
	}

	public static final Point2D[] transform3Dto2D(Point3D p) {
		double[] imageCoord = new double[4];
		double[] R1 = new double[] {p.x + T[0] / 2,p.y + T[1] / 2,p.z + T[2] / 2};
		double[] r1 = new double[] {(R1[0] / R1[2]) / CxL,(R1[1] / R1[2]) / CyL};
		double[] r2 = new double[] {r1[0] * (1.0 + KL * (sq(r1[0]) + sq(r1[1]))) + C0[0],r1[1] * (1.0 + KL * (sq(r1[0]) + sq(r1[1]))) + C0[1]};
		imageCoord[0] = r2[0];
		imageCoord[0] += -4.72664 + 0.00264009 * r2[0] + 3.68547E-7 * sq(r2[0]) - 0.003594 * r2[1] + 4.59175E-7 * r2[0] * r2[1] + 5.21369E-6 * sq(r2[1]);
		imageCoord[1] = r2[1];
		imageCoord[1] += 9.60826 - 0.00203106 * r2[0] - 3.99045E-6 * sq(r2[0]) - 0.0121255 * r2[1] + 5.57403E-6 * r2[0] * r2[1] + 1.79152E-6 * sq(r2[1]);
		R1 = new double[] {p.x - T[0] / 2,p.y - T[1] / 2,p.z - T[2] / 2};
		R1 = rotate(R1[0], R1[1], R1[2], Q[0], Q[1], Q[2], Q[3]);
		r1 = new double[] {(R1[0] / R1[2]) / CxR,(R1[1] / R1[2]) / CyR};
		r2 = new double[] {r1[0] * (1.0 + KR * (r1[0] * r1[0] + r1[1] * r1[1])) + C0[0],r1[1] * (1.0 + KR * (r1[0] * r1[0] + r1[1] * r1[1])) + C0[1]};
		imageCoord[2] = r2[0];
		imageCoord[2] += -11.058 + 0.0103577 * r2[0] - 4.38807E-6 * sq(r2[0]) + 0.0126752 * r2[1] + 7.04229E-7 * r2[0] * r2[1] - 5.02606E-6 * sq(r2[1]);
		imageCoord[3] = r2[1];
		imageCoord[3] += 1.05562 + 0.0044762 * r2[0] - 1.55134E-6 * sq(r2[0]) + 0.00646057 * r2[1] - 9.63543E-6 * r2[0] * r2[1] - 3.60981E-6 * sq(r2[1]);
		return new Point2D[] {new Point2D(imageCoord[0], imageCoord[1]),new Point2D(imageCoord[2], imageCoord[3])};
	}

	private static final double sq(double x) {
		return x * x;
	}
}

class Random {
	private static final long mask0 = 0x80000000L;
	private static final long mask1 = 0x7fffffffL;
	private static final long[] mult = new long[] {0,0x9908b0dfL};
	private final long[] mt = new long[624];
	private int idx = 0;

	Random(long seed) {
		init(seed + 9999);
	}

	private void init(long seed) {
		mt[0] = seed & 0xffffffffl;
		for (int i = 1; i < 624; i++) {
			mt[i] = 1812433253l * (mt[i - 1] ^ (mt[i - 1] >>> 30)) + i;
			mt[i] &= 0xffffffffl;
		}
	}

	private void generate() {
		for (int i = 0; i < 227; i++) {
			long y = (mt[i] & mask0) | (mt[i + 1] & mask1);
			mt[i] = mt[i + 397] ^ (y >> 1) ^ mult[(int) (y & 1)];
		}
		for (int i = 227; i < 623; i++) {
			long y = (mt[i] & mask0) | (mt[i + 1] & mask1);
			mt[i] = mt[i - 227] ^ (y >> 1) ^ mult[(int) (y & 1)];
		}
		long y = (mt[623] & mask0) | (mt[0] & mask1);
		mt[623] = mt[396] ^ (y >> 1) ^ mult[(int) (y & 1)];
	}

	private long rand() {
		if (idx == 0) generate();
		long y = mt[idx];
		idx = (idx + 1) % 624;
		y ^= (y >> 11);
		y ^= (y << 7) & 0x9d2c5680l;
		y ^= (y << 15) & 0xefc60000l;
		return y ^ (y >> 18);
	}

	int nextInt(int n) {
		return (int) (rand() % n);
	}

	double nextDouble() {
		return (rand() + 0.5) * (1.0 / 4294967296.0);
	}
}

class Geom {
	static double angle(Point2D pc, Point2D p) {
		double ang = Math.atan2(p.y - pc.y, p.x - pc.x);
		if (ang < 0) ang += Math.PI * 2;
		return ang;
	}

	static List<Point2D> convexHull(List<Point2D> p) {
		if (p == null) return null;
		if (p.size() < 3) return null;
		Point2D base = p.get(0);
		for (Point2D curr : p) {
			if (curr.y < base.y || (curr.y == base.y && curr.x > base.x)) {
				base = curr;
			}
		}
		List<Point2D> hull = new ArrayList<Point2D>();
		hull.add(base);
		double prevAng = -1;
		while (true) {
			Point2D next = null;
			double ang = Math.PI * 3;
			Point2D last = hull.get(hull.size() - 1);
			for (Point2D curr : p) {
				if (curr.equals(last)) continue;
				double ca = angle(last, curr);
				if (ca <= ang && ca >= prevAng) {
					ang = ca;
					next = curr;
				}
			}
			prevAng = ang;
			if (next == null || next.equals(base)) break;
			hull.add(next);
		}
		return hull;
	}
}

class Position implements Comparable<Position> {
	final double val;
	final double[] pos;

	public int compareTo(Position o) {
		return Double.compare(o.val, val);
	}

	public Position(double val, double[] pos) {
		this.val = val;
		this.pos = pos;
	}
}

class Beam<T extends Comparable<T>> {
	private int beamWidth;
	private Object[] items;
	private Object[] aux;
	private int size;

	Beam(int beamWidth) {
		this.beamWidth = beamWidth;
		items = new Object[beamWidth];
		aux = new Object[beamWidth];
	}

	void setWidth(int beamWidth) {
		this.beamWidth = beamWidth;
		if (items.length < beamWidth) {
			items = Arrays.copyOf(items, beamWidth);
			aux = new Object[beamWidth];
		}
		if (size > beamWidth) size = beamWidth;
	}

	@SuppressWarnings("unchecked")
			boolean add(T item) {
		if (size >= beamWidth && ((Comparable<T>) items[beamWidth - 1]).compareTo(item) <= 0) return false;
		int pos = Arrays.binarySearch(items, 0, size, item);
		if (pos < 0) pos = -pos - 1;
		else if (items[pos].equals(item)) return false;
		if (pos >= beamWidth) return false;
		if (size < beamWidth) size++;

		System.arraycopy(items, pos, aux, 0, size - pos - 1);
		System.arraycopy(aux, 0, items, pos + 1, size - pos - 1);
		items[pos] = item;
		return true;
	}

	@SuppressWarnings("unchecked")
			T get(int idx) {
		return (T) items[idx];
	}

	@SuppressWarnings("unchecked")
			T last() {
		return (T) items[size - 1];
	}

	int size() {
		return size;
	}

	void clear() {
		Arrays.fill(items, 0, size, null);
		size = 0;
	}
}

class ImgViewer extends JFrame {
	private static final long serialVersionUID = -5820105568092949073L;
	private static final Object lock = new Object();
	private static int off = 0;

	public ImgViewer(final BufferedImage img) {
		JPanel imagePanel = new JPanel() {
			private static final long serialVersionUID = -7037900892428152902L;

			public void paintComponent(Graphics g) {
				super.paintComponent(g);
				((Graphics2D) g).drawImage(img, 0, 0, 800, 600, null);
			}
		};
		getContentPane().add(imagePanel);
		synchronized (lock) {
			setBounds(off, off, 820, 660);
			off += 48;
			if (off > 300) off = 0;
		}
		setDefaultCloseOperation(JFrame.DISPOSE_ON_CLOSE);
		setVisible(true);
	}
}
