package p;
/* 
 * These imports are needed only in local testing. Commented out for submissions
 * as the TopCoder sandbox doesn't allow file operations. Likewise, several 
 * testing methods near the end of this class are commented out, they are
 * clearly marked as such.
 */
//import java.awt.image.BufferedImage;
//import java.io.File;
//import java.io.FileOutputStream;
//import javax.imageio.ImageIO;

import java.text.DecimalFormat;
import java.text.DecimalFormatSymbols;
import java.util.Arrays;
import java.util.HashSet;
import java.util.List;
import java.util.Set;
import java.util.Vector;

public class RobonautVision {
	private static final long TIME_LIMIT = 85000;

	private static boolean debug = true;
	public static String info;
	
	private static int W = 1600;
	private static int H = 1200;
	// small version of images, everything is scaled down by a factor of 2
	private static int scale = 2;
	private static int Ws = W / scale;
	private static int Hs = H / scale;
	
	private Image[] originalImages = new Image[2];
	private Image[] filteredImages = new Image[2];
	private Image[] smallImages = new Image[2];
	private double[][][] edgeMaps = new double[2][][];
	private double[][][] heatMaps = new double[2][][];
	private double[] heatSums = new double[2];
	
	private static Vertex[] originalVertices = null;
    private static Triangle[] originalFaces = null;
    private double modelD; // diameter of model
	public Point modelC; // center point of model
    
    private static int RR = 10; // rotation resolution
    private static double[] cosRR = new double[RR];
    private static double[] sinRR = new double[RR];
    static {
    	for (int i = 0; i < RR; i++) {
    		double x = 2 * Math.PI / RR * (i + 0.5);
			cosRR[i] = Math.cos(x);
			sinRR[i] = Math.sin(x);
		}
    }
    // max no. of contour points per orientation
    private static int CONTOUR_POINT_COUNT = 100;
    /*
     * Note that the first index of the contourPoints array is not used, only
     * contourPoints[0] is filled in. I planned to use more data (see description)
     * but didn't have the time.
     */
 	public Point[][][][][] contourPoints = new Point[9][RR][RR][RR][CONTOUR_POINT_COUNT];
 	private ContourFollower cf = new ContourFollower();
	
    private static int testCnt = 0;
    private long startTime, time;

    private Placement bestPlacement;
    private double bestScore;
    /*
     * contourPonts and bestRotIndexes are public so that I can add some extra
     * info to display in the visualization tool 
     */
    public int[] bestRotIndexes = new int[4];
    
    private Transform transform = new Transform();
   
	public int trainingModel(String[] plyData) {
		parseModel(plyData);
		simplify();
		return 1;
	}

	public int doneTraining() {
		return 1;
	}
		 
	public int trainingPair(int[] leftImage, int[] rightImage, double[] groundTruth) {
		return 1; // no training is done, we work only by the wireframe data
	}
		 
	public double[] testingPair(int[] leftImage, int[] rightImage) {
		startTime = System.currentTimeMillis();
		testCnt++;
		setupImages(leftImage, 0);
		setupImages(rightImage, 1);
		time = System.currentTimeMillis() - startTime;
		log("Setup done: " + f((double)time / 1000));
		// this is where the work is done:
		scan(); 
		time = System.currentTimeMillis() - startTime;
		log("Time: " + f((double)time / 1000) + ", iterations: " + iterations + ", f: " + f((double)iterations*1000/time));

		if (bestPlacement == null) {
			Placement dummy = new Placement(0,0,800, 1,0,0,0);
			return dummy.data;
		}

		// transform back from model center based coords to original system
		// real_pos = placement - rot(modelC)
		double[] displacement = transform.rotate(modelC.x, modelC.y, modelC.z, 
				bestPlacement.data[3], bestPlacement.data[4], bestPlacement.data[5], bestPlacement.data[6]);
		bestPlacement.data[0] -= displacement[0];
		bestPlacement.data[1] -= displacement[1];
		bestPlacement.data[2] -= displacement[2];
		
		return bestPlacement.data;
	}
	
	private int iterations = 0; // just for debugging
	
	private void scan() {
		double score = 0;
		double bestLeftScore = -1;
		bestScore = -1;
		bestPlacement = null;
		Placement p;
		int[] rotIndexes = new int[4];
		iterations = 0;
		
		// try a lot of placements and pick the best scoring one
		for (double x = -300; x <= 300; x += 20) { 
			for (double y = -200; y <= 200; y += 20) {
				for (double z = 700; z < 1000; z *= 1.08) {
					// the following triple loop should uniformly sample the 
					// rotation space
					for (int x0i = 0; x0i < RR; x0i++) {
						double x0 = (double)1 / RR * (x0i + 0.5);
						rotIndexes[1] = x0i;
						double r1 = Math.sqrt(1 - x0);
						double r2 = Math.sqrt(x0);
						for (int x1i = 0; x1i < RR; x1i++) {
							rotIndexes[2] = x1i;
							for (int x2i = 0; x2i < RR; x2i++) {
								if (cosRR[x1i] < 0) continue;
								rotIndexes[3] = x2i;
								double q0 = cosRR[x2i] * r2;
								double q1 = sinRR[x1i] * r1;
								double q2 = cosRR[x1i] * r1;
								double q3 = sinRR[x2i] * r2;
								p = new Placement(x,y,z, q0,q1,q2,q3);
								rotIndexes[0] = 0;
								double leftScore = place(p, 0, rotIndexes);
								double rightScore = 0;
								/*
								 * To save time we check the right eye only if 
								 * the left eye scored high enough.
								 */
								if (score > bestLeftScore * 0.7) {
									rightScore = place(p, 1, rotIndexes);
								}
								score = leftScore + rightScore;
								String best = "";
								if (score > bestScore) {
									bestScore = score;
									bestLeftScore = leftScore;
									bestPlacement = p;
									bestRotIndexes[0] = 0;
									bestRotIndexes[1] = x0i;
									bestRotIndexes[2] = x1i;
									bestRotIndexes[3] = x2i;
									best = "*";
								}
								flog(best + "\t"+ f(score) + " (" + f(leftScore) + ", " + f(rightScore) + ") " + p.toString());
								iterations++;
								if (iterations % 5000 == 0) {
									time = System.currentTimeMillis() - startTime;
									if (time > TIME_LIMIT) {
										log("Aborting at x="+x+" y="+y+" z="+z);
										return;
									}
								}
							}
						}
					}
				}
			}
		}
	}

	// try a placement on one of the images, return score
	private double place(Placement pos, int eye, int[] rotIndexes) {
		int hit = 0;
		int cntInImage = 0;
		int cntAll = 0;
		
		Point[] contourPs = contourPoints[rotIndexes[0]][rotIndexes[1]][rotIndexes[2]][rotIndexes[3]];
		// count how many of the contour points (given the current orientation)
		// are on image edges 
		for (Point point: contourPs) {
			if (point == null) break;
			cntAll++;
            double[] p = transform.rotate(
            		point.x, point.y, point.z,  pos.data[3], pos.data[4], pos.data[5], pos.data[6]);
            Point ray = new Point(p[0] + pos.data[0], p[1] + pos.data[1], p[2] + pos.data[2]);
            double[] p2d = transform.transform3Dto2D(eye, ray.x, ray.y, ray.z);
            int x = (int)(p2d[eye*2] + 0.5) / scale;
            int y = (int)(p2d[eye*2+1] + 0.5) / scale;
            if (x < 0 || x >= Ws || y < 0 || y >= Hs) continue;
            
            // we could use edge magnitude but it doesn't improve the recognition
            // so we take either zero or not
            double e = edgeMaps[eye][x][y];
            if (e != 0) hit++;
            cntInImage++;
            // trying to speed up by breaking early, doesn't work
            if (cntInImage == 5 && hit == 0) {
            	//return 0; 
            }
        }
		if (cntAll == 0) return 0;
		if (cntInImage < cntAll / 10) return 0; // when most of the object is out of the image
		double score = (double)hit / cntAll;
		return score;
	}
	
	// This if for debugging only, used from the visualizer
	public int[] getBestRotIndexes(Placement p) {
		//Conjugate
        double pq0 = p.data[3];
        double pq1 = -p.data[4];
        double pq2 = -p.data[5];
        double pq3 = -p.data[6];
        double len = pq0*pq0+pq1*pq1+pq2*pq2+pq3*pq3;
        pq0 /= len; pq1 /= len; pq2 /= len; pq3 /= len; 
        
		double minDiff = Double.MAX_VALUE;
		int[] best = new int[4];
		for (int x0i = 0; x0i < RR; x0i++) {
			double x0 = (double)1 / RR * (x0i + 0.5);
			double r1 = Math.sqrt(1 - x0);
			double r2 = Math.sqrt(x0);
			for (int x1i = 0; x1i < RR; x1i++) {
				if (cosRR[x1i] < 0) continue;
				for (int x2i = 0; x2i < RR; x2i++) {
					double q0 = cosRR[x2i] * r2;
					double q1 = sinRR[x1i] * r1;
					double q2 = cosRR[x1i] * r1;
					double q3 = sinRR[x2i] * r2;
					
					double diff = (p.data[3] - q0) * (p.data[3] - q0);
					diff += (p.data[4] - q1) * (p.data[4] - q1);
					diff += (p.data[5] - q2) * (p.data[5] - q2);
					diff += (p.data[6] - q3) * (p.data[6] - q3);
					
					if (diff < minDiff) {
						minDiff = diff;
						best = new int[]{0, x0i, x1i, x2i};
					}
				}
			}
		}
		return best;
	}

	////////////////////////////////////////////////////
	
	private void setupImages(int[] ints, int eye) {
		Image img = new Image(W, H);
		originalImages[eye] = img;
		for (int i = 0; i < W*H; i++) {
			int p = ints[i];
			int x = i % W;
			int y = i / W;
			int r = (p >> 16) & 0xff;
			int g = (p >> 8) & 0xff;
			int b = (p >> 0) & 0xff;
			img.rs[x][y] = r;
			img.gs[x][y] = g;
			img.bs[x][y] = b;
		}
		Image filtered = medianFilterWhereNoisy(img);
		filteredImages[eye] = filtered;
		Image small = downScale(filtered);
		smallImages[eye] = small;
		
		String eyeS = eye == 0 ? "left" : "right";
		makeHeatMap(eye, smallImages[eye]);
		BwImage heatImage = new BwImage(Ws, Hs);
		heatImage.data = heatMaps[eye];
		saveImage("heat-" + info + "-" + testCnt + "-" + eyeS, heatImage); // debug
		
		makeEdgeMap(eye, heatImage);
		BwImage edgeImage = new BwImage(Ws, Hs);
		edgeImage.data = edgeMaps[eye];
		saveImage("edge-" + info + "-" + testCnt + "-" + eyeS, edgeImage); // debug
	}
	
	/*
	 * Find the 'hot' areas of an image. These are the parts that look interesting
	 * because they differ from the rest of the image. For each pixel we calculate
	 * a heat score: how different the colour is from the neighbourhood. Difference
	 * is taken as the minimum of average similarity in the 4 main directions. 
	 */
	private void makeHeatMap(int eye, Image image) {
		int w = image.w; int h = image.h;
		double[][] map = new double[w][h];
		double max = 0;
		for (int i = 0; i < w; i++) {
			for (int j = 0; j < h; j++) {
				double diff = Double.MAX_VALUE;
				double d;
				d = getDiffOnLine(i, j, 1, 0, image); diff = Math.min(diff, d);
				d = getDiffOnLine(i, j, -1, 0, image); diff = Math.min(diff, d);
				d = getDiffOnLine(i, j, 0, 1, image); diff = Math.min(diff, d);
				d = getDiffOnLine(i, j, 0, -1, image); diff = Math.min(diff, d);
				map[i][j] = diff;
				max = Math.max(max, diff);
			}
		}
		// normalize heat map to have max=1
		double sum = 0;
		if (max > 0) {
			for (int i = 0; i < w; i++) for (int j = 0; j < h; j++) {
				map[i][j] /= max;
				sum += map[i][j];
			}
		}
		heatMaps[eye] = map;
		heatSums[eye] = sum;
	}

	/*
	 * Go max width/2 steps in a given direction from point (i,j) and return the
	 * average colour difference on this line. For points on large, boring areas
	 * this value will be small. 
	 */
	private double getDiffOnLine(int i, int j, int dx, int dy, Image image) {
		int w = image.w; int h = image.h;
		int maxCnt = Math.min(w, h) / 2;
		int r0 = image.rs[i][j];
		int g0 = image.gs[i][j];
		int b0 = image.bs[i][j];
		int cnt = 0;
		int diff = 0;
		i += dx; j += dy;
		while (i >= 0 && i < w && j >= 0 && j < h && cnt < maxCnt) {
			int dr = image.rs[i][j] - r0;
			int dg = image.gs[i][j] - g0;
			int db = image.bs[i][j] - b0;
			int d = dr*dr + dg*dg + db*db;
			diff += d;
			cnt++;
			i += dx;
			j += dy;
		}
		if (cnt == 0) return Double.MAX_VALUE;
		double ret = (double)diff / cnt;
		return ret;
	}

	// Create a smaller version by doing a scale x scale median filtering
	private Image downScale(Image img) {
		Image ret = new Image(Ws, Hs);
		int n = scale*scale;
		int[] rs = new int[n];
		int[] gs = new int[n];
		int[] bs = new int[n];		
		for (int i = 0; i < Ws; i++) {
			for (int j = 0; j < Hs; j++) {
				int cnt = 0;
				for (int x = 0; x < scale; x++) {
					for (int y = 0; y < scale; y++) {
						rs[cnt] = img.rs[i*scale + x][j*scale + y];
						gs[cnt] = img.gs[i*scale + x][j*scale + y];
						bs[cnt] = img.bs[i*scale + x][j*scale + y];
						cnt++;
					}
				}
				Arrays.sort(rs); Arrays.sort(gs); Arrays.sort(bs); 
				ret.rs[i][j] = rs[cnt/2];
				ret.gs[i][j] = gs[cnt/2];
				ret.bs[i][j] = bs[cnt/2];
			}
		}
		return ret;
	}

	/*
	 * Remove salt and pepper noise from points where it is present. If any
	 * of the RGB components is 0 or 255 then replace the colour value with
	 * the 5x5 median of the neighbours.
	 */
	private Image medianFilterWhereNoisy(Image img) { 
		Image ret = new Image(img.w, img.h);
		int d = 2;
		int n = (2*d+1)*(2*d+1);
		int[] rs = new int[n];
		int[] gs = new int[n];
		int[] bs = new int[n];		
		for (int i = d; i < img.w-d; i++) {
			for (int j = d; j < img.h-d; j++) {
				boolean ok = true;
				if (img.rs[i][j] == 0 || img.rs[i][j] == 255) ok = false;
				else if (img.gs[i][j] == 0 || img.gs[i][j] == 255) ok = false;
				else if (img.bs[i][j] == 0 || img.bs[i][j] == 255) ok = false;
				if (ok) {
					ret.rs[i][j] = img.rs[i][j];
					ret.gs[i][j] = img.gs[i][j];
					ret.bs[i][j] = img.bs[i][j];
					continue;
					
				}
				
				int cnt = 0;
				for (int x = i-d; x <= i+d; x++) {
					for (int y = j-d; y <= j+d; y++) {
						rs[cnt] = img.rs[x][y];
						gs[cnt] = img.gs[x][y];
						bs[cnt] = img.bs[x][y];
						cnt++;
					}
				}
				Arrays.sort(rs); Arrays.sort(gs); Arrays.sort(bs); 
				ret.rs[i][j] = rs[cnt/2];
				ret.gs[i][j] = gs[cnt/2];
				ret.bs[i][j] = bs[cnt/2];
			}
		}
		return ret;
	}
	
	// Sobel edge detection on a black and white image
	private void makeEdgeMap(int eye, BwImage img) {
		double[][] ds = new double[img.w][img.h];
		double max = 0;
		for (int i = 1; i < img.w-1; i++) for (int j = 1; j < img.h-1; j++) {
			double dx = img.data[i+1][j-1]-img.data[i-1][j-1] + 
					2*(img.data[i+1][j]-img.data[i-1][j]) +
					img.data[i+1][j+1]-img.data[i-1][j+1];
			double dy = img.data[i-1][j+1]-img.data[i-1][j-1] + 
					2*(img.data[i][j+1]-img.data[i][j-1]) +
					img.data[i+1][j+1]-img.data[i+1][j-1];
			
			double d = Math.sqrt(dx*dx + dy*dy);
			ds[i][j] = d;
			max = Math.max(max, d);
		}
		/*
		 * Normalize to max = 0
		 * Also spread the edge image a bit: for each pixel if the value is less
		 * than half of its strongest neighbour then increase to that (half) value.
		 * Repeat this 10 times.
		 */
		if (max > 0) {
			for (int i = 1; i < img.w-1; i++) for (int j = 1; j < img.h-1; j++) {
				double e = ds[i][j] / max;
				if (e < 0.1) e = 0;
				ds[i][j] = e;
			}
			for (int k = 0; k < 10; k++) {
				double[][] tmp = new double[img.w][img.h];
				for (int i = 1; i < img.w-1; i++) for (int j = 1; j < img.h-1; j++) {
					double v = ds[i][j];
					double n = Math.max(ds[i-1][j], ds[i+1][j]);
					n = Math.max(n, ds[i][j-1]);
					n = Math.max(n, ds[i][j+1]);
					if (n > 2 * v) {
						tmp[i][j] = n / 2;
					}
					else {
						tmp[i][j] = v;
					}
				}
				ds = tmp;
			}
		}
		edgeMaps[eye] = ds;				
	}
	
	////////////////////////////////////////////////////////////

	public void parseModel(String[] plydata)
    {
		int size = plydata.length;
		int numVertex = 0;
        int numFaces = 0;
        int idx = 0;
        while (!plydata[idx].equals("end_header")) {
        	//log(idx + ": " + plydata[idx]);
            String[] tok = plydata[idx].split(" ");
            if (tok[0].equals("element")) {
                if (tok[1].equals("vertex")) {
                    numVertex = Integer.parseInt(tok[2]);
                } 
                else if (tok[1].equals("face")) {
                    numFaces = Integer.parseInt(tok[2]);
                }
            }
            idx++;
        }
        idx++;

        originalVertices = new Vertex[numVertex];
        for (int i=0;i<numVertex;i++) {
        	//if (i == 0) log(idx + ": " + plydata[idx]);
            String[] tok = plydata[idx++].split(" ");
            double x = Double.parseDouble(tok[0]);
            double y = Double.parseDouble(tok[1]);
            double z = Double.parseDouble(tok[2]);
            originalVertices[i] = new Vertex(i, new Point(x,y,z));
        }
        // Read faces
        List<Triangle> tList = new Vector<>();
        for (int i=0;i<numFaces;i++) {
        	//if (i == 0) log(idx + ": " + plydata[idx]);
        	if (idx == size) break;
            String[] tok = plydata[idx++].split(" ");
            Vertex[] originalVs = new Vertex[3];
            for (int j=0; j < 3; j++) {
            	int n = Integer.parseInt(tok[1+j]);
            	originalVs[j] = originalVertices[n];
            }
            tList.add(new Triangle(i, originalVs));
        }
        originalFaces = tList.toArray(new Triangle[0]);
    }
	
	/*
	 * Do an uniform sampling of the rotation space and for each orientation
	 * determine some points that are on the contour of the image.
	 */
	private void simplify() {
		startTime = System.currentTimeMillis();
		double minx = Double.MAX_VALUE;
		double maxx = -Double.MAX_VALUE;
		double miny = Double.MAX_VALUE;
		double maxy = -Double.MAX_VALUE;
		double minz = Double.MAX_VALUE;
		double maxz = -Double.MAX_VALUE;
		for (Vertex v: originalVertices) {
			minx = Math.min(v.position.x, minx);
			maxx = Math.max(v.position.x, maxx);
			miny = Math.min(v.position.y, miny);
			maxy = Math.max(v.position.y, maxy);
			minz = Math.min(v.position.z, minz);
			maxz = Math.max(v.position.z, maxz);
		}
		double cx = (maxx + minx) / 2;
		double cy = (maxy + miny) / 2;
		double cz = (maxz + minz) / 2;
		modelC = new Point(cx, cy, cz);
		modelD = Math.max(maxx - minx, maxy - miny);
		modelD = Math.max(modelD, maxz - minz);
		
		// update the model points so that the model is centered at (0,0,0)
		for (Vertex v: originalVertices) {
			v.position.x -= cx;
			v.position.y -= cy;
			v.position.z -= cz;
		}
		
		int edgeCnt = 1; // real edge ids start from 1!
		List<Edge> edges = new Vector<>(originalFaces.length * 3 / 2);
		edges.add(new Edge()); // just that id 1 will be a real edge
		for (Triangle t: originalFaces) {
			for (int i = 0; i < 3; i++) {
				Vertex v1 = t.vs[i];
				Vertex v2 = t.vs[(i+1)%3];
				if (v1.id > v2.id) continue;
				Edge e = new Edge();
				e.id = edgeCnt++;				 
				e.v1 = v1; e.v2 = v2;
				edges.add(e);
			}
		}
		
		// http://refbase.cvc.uab.es/files/PIE2012.pdf
		for (int x0i = 0; x0i < RR; x0i++) {
			double x0 = (double)1 / RR * (x0i + 0.5);
			double r1 = Math.sqrt(1 - x0);
			double r2 = Math.sqrt(x0);
			for (int x1i = 0; x1i < RR; x1i++) {
				double x1 = (double)1 / RR * (x1i + 0.5);
				double theta1 = 2 * Math.PI * x1;
				for (int x2i = 0; x2i < RR; x2i++) {
					double x2 = (double)1 / RR * (x2i + 0.5);
					double theta2 = 2 * Math.PI * x2;
					double q0 = Math.cos(theta2) * r2;
					double q1 = Math.sin(theta1) * r1;
					double q2 = Math.cos(theta1) * r1;
					double q3 = Math.sin(theta2) * r2;
					Placement pos = new Placement(0,0,600, q0,q1,q2,q3);
					fillContourPoints(edges, pos, 0, x0i, x1i, x2i);
				}
			}
		}
		time = System.currentTimeMillis() - startTime;
		log("Simplify done " + f(time / 1000.0));
	}
	
	/*
	 * Create an image by drawing every edge (with a given orientation of the
	 * object) on an imaginary canvas then find the contour of the image. 
	 */
	private void fillContourPoints(List<Edge> edges, Placement pos, int cell, int x0i, int x1i, int x2i) {
		int n = originalVertices.length;
		boolean debugImage = false;
		if (x2i == 0 && x0i == 0) {
			//debugImage = true;
		}
		double[][] p2d = new double[n][2];
		double minx = Double.MAX_VALUE;
		double maxx = -Double.MAX_VALUE;
		double miny = Double.MAX_VALUE;
		double maxy = -Double.MAX_VALUE;
		for (Vertex v: originalVertices) {
			double[] loc = transform.rotate(v.position.x, v.position.y, v.position.z, pos.data[3], pos.data[4], pos.data[5], pos.data[6]);
			double[] p = transform.transform3Dto2D(0, loc[0] + pos.data[0], loc[1] + pos.data[1], loc[2] + pos.data[2]);
			p2d[v.id][0] = p[0];
			p2d[v.id][1] = p[1];
			minx = Math.min(p[0], minx);
			maxx = Math.max(p[0], maxx);
			miny = Math.min(p[1], miny);
			maxy = Math.max(p[1], maxy);
		}
		double d = Math.max(maxx-minx, maxy-miny);
		for (int i = 0; i < n; i++) {
			p2d[i][0] = (p2d[i][0] - minx) / d;
			p2d[i][1] = (p2d[i][1] - miny) / d;
		}
		int margin = 4; 
		int size = 1000 + 2 * margin;
		int[][] img = new int[size][size];
		for (Edge e: edges) {
			if (e.id == 0) continue;
			int x1 = margin + (int)(p2d[e.v1.id][0] * 1000 + 0.5);
			int y1 = margin + (int)(p2d[e.v1.id][1] * 1000 + 0.5);
			int x2 = margin + (int)(p2d[e.v2.id][0] * 1000 + 0.5);
			int y2 = margin + (int)(p2d[e.v2.id][1] * 1000 + 0.5);
			line(x1,y1, x2,y2, img, e);
		}
		
		cf.followContour(img, size, size);
		
		Image iout = null;
		if (debugImage) {
			iout = new Image(size, size);
			for (int i = 0; i < size; i++) for (int j = 0; j < size; j++) {
				if (cf.labels[i][j] != 0) iout.gs[i][j] = 100;
				if (img[i][j] != 0) iout.bs[i][j] = 255;
			}
		}
		
		List<Point> contourPointList = new Vector<>(CONTOUR_POINT_COUNT);
		/*
		 * Divide the image into 20x20 cells and find a contour point within
		 * each cell. Find which edge the point belongs to, and get the 3D coordinates
		 * of the corresponding point of the target object.
		 */
		int N = 20;
		int cellSize = size / N;
		for (int i = 0; i < N; i++) for (int j = 0; j < N; j++) {
			NEXTCELL:
			for (int x = 0; x < cellSize; x++) for (int y = 0; y < cellSize; y++) {
				int px = i * cellSize + x; int py = j * cellSize + y;
				if (cf.labels[px][py] == 0) continue;
				int edgeId = img[px][py];
				if (edgeId != 0) {
					Edge e = edges.get(edgeId);
					int x1 = margin + (int)(p2d[e.v1.id][0] * 1000 + 0.5);
					int y1 = margin + (int)(p2d[e.v1.id][1] * 1000 + 0.5);
					int x2 = margin + (int)(p2d[e.v2.id][0] * 1000 + 0.5);
					int y2 = margin + (int)(p2d[e.v2.id][1] * 1000 + 0.5);
					double dx = x2 - x1;
					double dy = y2 - y1;
					if (dx == 0 && dy == 0) {
						contourPointList.add(e.v1.position);
					}
					else { // linear interpolation between the edge endpoints
						d = dx;
						double dp;
						if (Math.abs(dx) > Math.abs(dy)) {
							dp = px - x1;
						}
						else {
							d = dy; dp = py - y1;
						}
						double vx = e.v1.position.x + (e.v2.position.x - e.v1.position.x) * dp / d;
						double vy = e.v1.position.y + (e.v2.position.y - e.v1.position.y) * dp / d;
						double vz = e.v1.position.z + (e.v2.position.z - e.v1.position.z) * dp / d;
						contourPointList.add(new Point(vx, vy, vz));
					}
					if (debugImage)	{
						iout.bs[px][py] = 255; iout.rs[px][py] = 255;
					}
					break NEXTCELL;
				}
			}
		}

		if (debugImage) {
			saveImage("_lines-"+x0i+"-"+x1i+"-"+x2i, iout);
		}
		
		Point[] pArr = contourPointList.toArray(new Point[0]);
		/*
		 * If there are too many points then pick those that are closer to the 
		 * top of the image. The assumption is that there'll be stronger edges
		 * on the target image on the top, because of shadows on the bottom.
		 */
		Arrays.sort(pArr);
		for (int i = 0; i < Math.min(CONTOUR_POINT_COUNT, pArr.length); i++) {
			contourPoints[cell][x0i][x1i][x2i][i] = pArr[i];
		}
	}
	
	// draw a line between 2 points
	private void line(int x,int y, int x2, int y2, int[][] image, Edge e) {
		int w = x2 - x ;
	    int h = y2 - y ;
	    int dx1 = 0, dy1 = 0, dx2 = 0, dy2 = 0 ;
	    if (w<0) dx1 = -1 ; else if (w>0) dx1 = 1 ;
	    if (h<0) dy1 = -1 ; else if (h>0) dy1 = 1 ;
	    if (w<0) dx2 = -1 ; else if (w>0) dx2 = 1 ;
	    int longest = Math.abs(w) ;
	    int shortest = Math.abs(h) ;
	    if (!(longest>shortest)) {
	        longest = Math.abs(h) ;
	        shortest = Math.abs(w) ;
	        if (h<0) dy2 = -1 ; else if (h>0) dy2 = 1 ;
	        dx2 = 0 ;            
	    }
	    int numerator = longest >> 1 ;
	    for (int i=0;i<=longest;i++) {
	    	image[x][y] = e.id;
	    	numerator += shortest ;
	        if (!(numerator<longest)) {
	            numerator -= longest ;
	            x += dx1 ;
	            y += dy1 ;
	        } else {
	            x += dx2 ;
	            y += dy2 ;
	        }
	    }
	}
	
	/////////////////////////////////////////////////////////
	// Helper classes
	
	private class Edge {
		public int id;
		public Vertex v1;
		public Vertex v2;
	}
	
	public class Image {
		public int w,h;
		public int[][] rs; // rgb elements are in 0..255
		public int[][] gs;
		public int[][] bs;
		
		public Image(int w, int h) {
			this.w = w; this.h = h;
			rs = new int[w][h];
			gs = new int[w][h];
			bs = new int[w][h];
		}
	}
	
	public class BwImage { // data elements are in 0..1
		public int w,h;
		public double[][] data;
		
		public BwImage(int w, int h) {
			this.w = w; this.h = h;
			data = new double[w][h];
		}
	}
	
	public static class Point implements Comparable<Point> {
        double x,y,z;
        
        public Point(double x, double y, double z) {
            this.x = x;
            this.y = y;
            this.z = z;
        }
        
        public static Point cross(Point u, Point v) {
        	return new Point(u.y*v.z-u.z*v.y, u.z*v.x-u.x*v.z, u.x*v.y-u.y*v.x);
        }
        
        public static double dot(Point u, Point v) {
        	return u.x*v.x + u.y*v.y + u.z*v.z;
        }
        
        public double len() {
        	return Math.sqrt(x*x + y*y + z*z);
        }
        
        public double dist(Point p) {
        	return Math.sqrt((x-p.x)*(x-p.x) + (y-p.y)*(y-p.y) +(z-p.z)*(z-p.z));
        }
        
        public Point plus(Point p) {
        	return new Point(x + p.x, y + p.y, z + p.z);
        }
        
        public Point minus(Point p) {
        	return new Point(x - p.x, y - p.y, z - p.z);
        }
        
        public String toString() {
        	return x + " " + y + " " + z;
        }

		public void normalize() {
			double d = len();
			if (d > 0) {
				x /= d; y /= d; z /= d;
			}
		}

		public int compareTo(Point o) { // sort by y
			if (y < o.y) return -1;
			if (y > o.y) return 1;
			if (x < o.x) return -1;
			if (x > o.x) return 1;
			return 0;
		}
    }
	
	public static class Vertex {
		public Point position;
		public int id;
		public Set<Vertex> neighbours = new HashSet<>();
		public List<Triangle> faces = new Vector<>();
		
		public Vertex(int id, Point p) {
			this.id = id;
			position = p;
		}
		
		public String toString() {
        	return "(" + id + ") " + position.toString();
        }

		public int hashCode() {
			return id;
		}
		
		public boolean equals(Object o) {
			return id == ((Vertex)o).id;
		}
	}

    public static class Triangle {
    	public int id;
        public Vertex[] vs = new Vertex[3];
        public Point normal;
        
        public Triangle(int id, Vertex[] vs) {
        	this.id = id;
            this.vs = vs;
            computeNormal();
            
            for (Vertex v: vs) {
            	v.faces.add(this);
            	for (Vertex v2: vs) {
            		if (v.id != v2.id) v.neighbours.add(v2); 
            	}
            }
        }

		public void computeNormal() {
			Point p1 = vs[1].position.minus(vs[0].position);
            Point p2 = vs[2].position.minus(vs[0].position);
            normal = Point.cross(p1, p2);
            normal.normalize();
		}

		public boolean hasVertex(Vertex v) {
			for (Vertex w: vs) {
				if (w.equals(v)) return true;
			}
			return false;
		}
		
		public String toString() {
        	return vs[0].id + " " + vs[1].id + " " + vs[2].id;
        }
    }
    
    public static class Placement {
        // x,y,z;
        // qr,qi,qj,qk;
        double[] data = new double[7];
        public Placement(double x, double y, double z, double qr, double qi, double qj, double qk) {
            data[0] = x;
            data[1] = y;
            data[2] = z;
            data[3] = qr;
            data[4] = qi;
            data[5] = qj;
            data[6] = qk;
            normalize();
        }
        public Placement(double[] _dt) {
            this.data  = _dt;
            normalize();
        }
        public void normalize() {
            double sqlen = data[3]*data[3] + data[4]*data[4] + data[5]*data[5] + data[6]*data[6];
            sqlen = Math.sqrt(sqlen);
            data[3] /= sqlen;
            data[4] /= sqlen;
            data[5] /= sqlen;
            data[6] /= sqlen;
        }
        
        public String toString() {
        	StringBuilder sb = new StringBuilder();
        	for (double d: data) sb.append(f(d)).append(" ");
        	return sb.toString();
        }
    }

    public static class Transform {
        // vector from the center of the left camera to the center of the right camera
        public double[] T = new double[] {145.626654332161,1.65379695634088,-3.65966860066967};
        // quaternion parameters, correcting the right camera direction
        public double[] Q = new double[] {0.999985, 0.0000680463, -7.05536E-7, 0.00540286};
        // transition coefficients = tangent of half of the camera view angle divided by a mm-size of the pixel cell on the camera element.
        public double CxL = 0.000529185;
        public double CyL = 0.000528196;
        public double CxR = 0.000529527;
        public double CyR = 0.000528341;
        // Distortion coefficients
        public double KL = -2.41479E-8;
        public double KR = -2.21532E-8;
        // Coordinate Center
        public double[] C0 = new double[] {800, 600};

        public double[] crossProduct(double[] lhs, double[] rhs)
        {
            return new double[]{lhs[1]*rhs[2]-lhs[2]*rhs[1], lhs[2]*rhs[0]-lhs[0]*rhs[2], lhs[0]*rhs[1]-lhs[1]*rhs[0]};
        }

        public double[] rotate(double x, double y, double z, double qr, double qi, double qj, double qk)
        {
            double[] v = new double[] {x,y,z};
            double[] qvec = new double[]{qi, qj, qk};
            double[] uv = crossProduct(qvec, v);
            double[] uuv = crossProduct(qvec, uv);
            uv[0] *= 2.0 * qr;
            uv[1] *= 2.0 * qr;
            uv[2] *= 2.0 * qr;
            uuv[0] *= 2.0;
            uuv[1] *= 2.0;
            uuv[2] *= 2.0;
            return new double[] {v[0]+uv[0]+uuv[0], v[1]+uv[1]+uuv[1], v[2]+uv[2]+uuv[2]};
        }

        public double[] transform3Dto2D(double x, double y, double z)
        {
        	return transform3Dto2D(2, x, y, z);
        }
        
        /*
         * A version of the transform method that works on either of the eyes
         * or on both eyes. Just to speed things up.
         * Parameter eye: 0: left, 1: right, 2: both
         */
        public double[] transform3Dto2D(int eye, double x, double y, double z)
        {
            double[] imageCoord = new double[4];
            double[] R1, r1, r2;
            
            if (eye != 1) {
	            // Left Camera
	            //1. Correction on the camera position.
	            R1 = new double[] { x + T[0]/2, y + T[1]/2, z + T[2]/2 };
	            //2. 3D Coordinates into the pixels in ideal (undistorted) image with the coordinate origin in the middle of the image
	            r1 = new double[] {(R1[0]/R1[2])/CxL , (R1[1]/R1[2])/CyL};
	            //3. Distortion Applied and the coordinate origin is set to the image corner
	            r2 = new double[] {r1[0]*(1.0 + KL * (r1[0]*r1[0]+r1[1]*r1[1]))+C0[0],
	                                        r1[1]*(1.0 + KL * (r1[0]*r1[0]+r1[1]*r1[1]))+C0[1]};
	            //4. Finally, the quadratic-form fit is applied
	            imageCoord[0] = r2[0];
	            imageCoord[0] += -4.72664 + 0.00264009*r2[0] + 3.68547E-7*r2[0]*r2[0] - 0.003594*r2[1] +  4.59175E-7*r2[0]*r2[1] + 5.21369E-6*r2[1]*r2[1];
	            imageCoord[1] = r2[1];
	            imageCoord[1] += 9.60826 - 0.00203106*r2[0] - 3.99045E-6*r2[0]*r2[0] - 0.0121255*r2[1] +  5.57403E-6*r2[0]*r2[1] + 1.79152E-6*r2[1]*r2[1];
            }
            
            if (eye != 0) {
	            // Right Camera
	            //1. Correction on the camera position.
	            R1 = new double[] { x - T[0]/2, y - T[1]/2, z - T[2]/2 };
	            //2. Rotation correct between left and right camera
	            R1 = rotate(R1[0], R1[1], R1[2], Q[0], Q[1], Q[2], Q[3]);
	            //3. 3D Coordinates into the pixels in ideal (undistorted) image with the coordinate origin in the middle of the image
	            r1 = new double[] {(R1[0]/R1[2])/CxR , (R1[1]/R1[2])/CyR};
	            //4. Distortion Applied and the coordinate origin is set to the image corner
	            r2 = new double[] {r1[0]*(1.0 + KR * (r1[0]*r1[0]+r1[1]*r1[1]))+C0[0],
	                               r1[1]*(1.0 + KR * (r1[0]*r1[0]+r1[1]*r1[1]))+C0[1]};
	            //5. Finally, the quadratic-form fit is applied
	            imageCoord[2] = r2[0];
	            imageCoord[2] += -11.058 + 0.0103577*r2[0] - 4.38807E-6*r2[0]*r2[0] + 0.0126752*r2[1] +  7.04229E-7*r2[0]*r2[1] - 5.02606E-6*r2[1]*r2[1];
	            imageCoord[3] = r2[1];
	            imageCoord[3] += 1.05562 + 0.0044762*r2[0] - 1.55134E-6*r2[0]*r2[0] + 0.00646057*r2[1] -  9.63543E-6*r2[0]*r2[1] - 3.60981E-6*r2[1]*r2[1];
            }
            
            return imageCoord;
        }
    }
    
    /*
     * Takes an image and finds the contour points. Results are in the labels
     * array after followContour() is called.
     */
    public class ContourFollower {
    	public int[][] image;
    	public int[][] labels;
    	
    	// assume background = 0
    	public void followContour(int[][] img, int w, int h) {
    		this.image = img;
    		labels = new int[w][h];
    		int startX = 0; 
    		int startY = 0;
    		int firstX;
    		int firstY;
    		int prevX; 
    		int prevY;
    		int currX; 
    		int currY;
    		
    		FOUND:
    		for (int j = 0; j < h; j++) {
    			for (int i = 0; i < w; i++) {
    	    		if (image[i][j] != 0) {
    					startX = i; startY = j; break FOUND;
    				}
    			}
    		}
    		
    		int[] next = nextStep(startX, startY, 0);
    		int dir = next[2];
    		labels[startX][startY] = 1;
    		prevX = startX; prevY = startY;
    		currX = firstX = next[0];
    		currY = firstY = next[1];
    		
    		int steps = 0;
    		while (true) {
    			labels[currX][currY] = 1; // mark as contour
    			dir = (dir + 6) % 8;
    			next = nextStep(currX, currY, dir);
    			dir = next[2];
    			prevX = currX;  prevY = currY;	
    			currX = next[0]; currY = next[1]; 
    			
    			if (prevX == startX && prevY == startY && currX == firstX && currY == firstY) {
    				break;
    			}
    			steps++;
    			if (steps > 10 * (w+h)) break;
    		}
    	}
    	
    	private final int[][] DXY = {
    		{ 1,0}, { 1, 1}, {0, 1}, {-1, 1}, {-1,0}, {-1,-1}, {0,-1}, { 1,-1}};
    	
    	int[] nextStep(int pX, int pY, int dir) { //[x, y, dir]
    		int[] ret = new int[3];
    		for (int i = 0; i < 7; i++) {
    			int x = pX + DXY[dir][0];
    			int y = pY + DXY[dir][1];
    			if (image[x][y] == 0) {
    				labels[x][y] = -1; // mark as just outside of contour
    				dir = (dir + 1) % 8;
    			} 
    			else {
    				ret[0] = x; ret[1] = y; ret[2] = dir;
    				break;
    			}
    		}
    		return ret;
    	}
    }
    
    public static void log(String s) {
    	System.out.println(s);
    }
	private static String[] spaces = new String[40];
    static {
    	for (int i = 0; i < 40; i++) {
    		String s = "";
    		for (int j = 0; j < i; j++) s += " ";
    		spaces[i] = s;
    	}
    }
    
    public static void log(int tab, String s) {
    	if (s == null) s = "";
    	String[] lines = s.split("\n");
    	for (String line: lines) System.out.println(spaces[tab] + line);
    }
    
    // Commented out for submissions.
    //private static FileOutputStream out;
    public void flog(String s) {
    	if (!debug) return;
//    	try {
//			if (out == null) out = new FileOutputStream("../out/out.log");
//			out.write((s + "\n").getBytes());
//			out.flush();
//		} catch (Exception e) {
//			e.printStackTrace();
//		}
    }
    
    public void saveImage(String name, Image image) {
		if (!debug) return;
		if (!name.endsWith(".png")) name += ".png";
	    // Commented out for submissions.
//		try {
//			BufferedImage bi = new BufferedImage(image.w, image.h, BufferedImage.TYPE_INT_RGB);
//            for (int x = 0; x < image.w; x++) {
//				for (int y = 0; y < image.h; y++) {
//					int c = (image.rs[x][y] << 16) | (image.gs[x][y] << 8) | (image.bs[x][y]);
//					bi.setRGB(x, y, c);
//				}
//            }
//            ImageIO.write(bi, "PNG", new File("../out/" + name));
//            
//		} catch (Exception e) {
//			e.printStackTrace();
//		}
	}
    
    private void saveImage(String name, BwImage image) {
		if (!debug) return;
		if (!name.endsWith(".png")) name += ".png";
	    // Commented out for submissions.
//		try {
//			BufferedImage bi = new BufferedImage(image.w, image.h, BufferedImage.TYPE_INT_RGB);
//            for (int x = 0; x < image.w; x++) {
//				for (int y = 0; y < image.h; y++) {
//					int v = (int)(255 * image.data[x][y]);
//					int c = (v << 16) | (v << 8) | (v);
//					bi.setRGB(x, y, c);
//				}
//            }
//            ImageIO.write(bi, "PNG", new File("../out/" + name));
//            
//		} catch (Exception e) {
//			e.printStackTrace();
//		}
	}

    private static DecimalFormat df; 
	static {
		String pattern = "0.###";
		df = new DecimalFormat(pattern);
		DecimalFormatSymbols dfs = new DecimalFormatSymbols();
		dfs.setDecimalSeparator('.');
		df.setDecimalFormatSymbols(dfs);
	}

	public static String f(double d) {
		return df.format(d);
	}
}
