package codeprocessor;

/** 
 * DynamicEditor.java
 *
 * Description:		This class is a specialized proxy for the real plugin class
 * @author			Tim "Pops" Roberts (troberts@bigfoot.com)
 * @version			1.0
 */

import java.lang.reflect.Method;
import java.util.*;

import com.topcoder.client.contestant.ProblemComponentModel;
import com.topcoder.shared.language.Language;
import com.topcoder.shared.problem.Renderer;

class DynamicCodeProcessor {

	// Editor Object and it's methods
	private Object processor = null;	
	private HashMap methodCache = new HashMap();
	
	// Functions Constants
	private final static String CONFIGURE			= "configure";
	private final static String PREPROCESS			= "preProcess";
	private final static String POSTPROCESS			= "postProcess";
	private final static String GETUSERDEFINEDTAGS	= "getUserDefinedTags";
	
	private String codeProcessorName;

	public DynamicCodeProcessor(String codeProcessor) throws InstantiationError {
		
		if (codeProcessor==null) throw instantiationError("Null CodeProcessor", "Null CodeProcessor");
		
		if(codeProcessor.equals(EntryPoint.class.getName())) {
			throw instantiationError(codeProcessor, "You cannot embed the code processor within itself");
		}
		
						
		Class codeProcessorClass;
		try {
			codeProcessorClass = Class.forName(codeProcessor);
		} catch(ClassNotFoundException e) {
			throw instantiationError(codeProcessor, e.toString());
		}

		// Store all the methodCache into the cache
		methodCache.put(PREPROCESS, getMethod(codeProcessorClass, PREPROCESS, new Class[] {String.class, ProblemComponentModel.class, Language.class, Renderer.class}));
		methodCache.put(POSTPROCESS, getMethod(codeProcessorClass, POSTPROCESS, new Class[] {String.class, Language.class}));
		methodCache.put(GETUSERDEFINEDTAGS, getMethod(codeProcessorClass, GETUSERDEFINEDTAGS, new Class[] {}));
		methodCache.put(CONFIGURE, getMethod(codeProcessorClass, CONFIGURE, new Class[] {}));

		//if(methodCache.get(PREPROCESS)==null) System.out.println("FYI - The preProcess method was not found in " + codeProcessor);
		//if(methodCache.get(POSTPROCESS)==null) System.out.println("FYI - The postProcess method was not found in " + codeProcessor);
	
		// Create the editor object
		try {
			processor = codeProcessorClass.newInstance();
		} catch (IllegalAccessException e) {
			throw instantiationError(codeProcessor, e.toString());
		} catch (InstantiationException e) {
			throw instantiationError(codeProcessor, e.toString());
		}
		
		codeProcessorName = codeProcessor;
		
	}
	
	public String getCodeProcessorName() {
		return codeProcessorName;
	}
	
	public boolean isConfigureDefined() {
		return methodCache.get(CONFIGURE)!=null;
	}
	
	public boolean isPreProcessDefined() {
		return methodCache.get(PREPROCESS)!=null;
	}
	
	public boolean isPostProcessDefined() {
		return methodCache.get(POSTPROCESS)!=null;
	}
	
	public boolean isUserDefinedTagsDefined() {
		return methodCache.get(GETUSERDEFINEDTAGS)!=null;
	}
	
	/* ---------------------------- INTERFACE methods -------------------------*/
	public String preProcess(String source, ProblemComponentModel component, Language language, Renderer renderer) { 
		try {
			if(methodCache.get(PREPROCESS)==null) return source;
			return (String)invokeMethod(PREPROCESS, new Object[] {source, component, language, renderer}); 
		} catch (ClassCastException e) {
			printBadRC(PREPROCESS, String.class.toString());
			return null;
		}
	}

	
	public String postProcess(String source, Language language) { 
		try {
			if(methodCache.get(POSTPROCESS)==null) return source;
			return (String)invokeMethod(POSTPROCESS, new Object[] {source, language}); 
		} catch (ClassCastException e) {
			printBadRC(POSTPROCESS, String.class.toString());
			return null;
		}
	}

	public Map getUserDefinedTags() { 
		try {
			if(methodCache.get(GETUSERDEFINEDTAGS)==null) return new HashMap();
			return (Map)invokeMethod(GETUSERDEFINEDTAGS, new Object[] {}); 
		} catch (ClassCastException e) {
			printBadRC(GETUSERDEFINEDTAGS, Map.class.toString());
			return null;
		}
	}

	public void configure() { 
		if(methodCache.get(CONFIGURE)==null) return;
		invokeMethod(CONFIGURE, new Object[] {}); 
	}

	/* ---------------------------- HELPER methods ----------------------------*/
	// Invokes a given method with the passed parameters
	private final Object invokeMethod(String methodName, Object[] parms) {
		if(methodCache==null || processor==null) return null;
		try {
			// Invoke the proper method with the passed parameters
			Method mthd = (Method)methodCache.get(methodName);
			return mthd==null ? null : mthd.invoke(processor, parms);
		} catch (Exception e) {
			System.err.println("Error invoking method " + methodName + "(" + (parms==null ? "" : Arrays.asList(parms).toString()) + ")");
			e.fillInStackTrace().printStackTrace();
			return null;
		}
	}
	
	// Adds the specified Method (or null if not found) to the method cache
	private final Method getMethod(Class pluginClass, String methodName, Class[] parms) {
		try {
			Method method = pluginClass.getMethod(methodName, parms);
			return method;
		} catch (NoSuchMethodException e) {
			return null;
		}
	}

	// Print the bad classpath message
	private final void printBadClassPath(String pluginName, String classPath) {
		StringBuffer str = new StringBuffer("The classpath ");
		str.append(classPath);
		str.append(" for the plugin ");
		str.append(pluginName);
		str.append(" threw a MalFormedURLException and will be ignored.");
		System.err.println(str.toString());
	}
	
	// Print the bad return code message
	private final void printBadRC(String methodName, String expected) {
		StringBuffer str = new StringBuffer("Method ");
		str.append(methodName);
		str.append(" did not return an object of type ");
		str.append(expected);
		str.append(".  Returned object ignored.");
		System.err.println(str.toString());
	}
	
	// Creates the NoSuchMethodError	
	private final NoSuchMethodError noSuchMethod(String methodName) {
		// Format the error message
		StringBuffer str = new StringBuffer("Required Method: ");
		str.append(methodName);
		str.append(" is not defined by the editor plugin.");
		
		// Write to console and throw error
		NoSuchMethodError error = new NoSuchMethodError(str.toString());
		error.fillInStackTrace().printStackTrace();
		return error;
	}
		 
	// Creates the InstantiationEror
	private final InstantiationError instantiationError(String className, String reason) {
		// Format the error message
		StringBuffer str = new StringBuffer("Cannot instantiate ");
		str.append(className);
		str.append(". ");
		str.append(reason);
		
		// Write to console and throw error
		InstantiationError error = new InstantiationError(str.toString());
		error.fillInStackTrace().printStackTrace();
		return error;
	}

}
